#include "torque_ver.h"
#include "console/simBase.h"

#include "console/consoleObject.h"
#include "console/consoleTypes.h"
#include "game/gameBase.h"
#include "sim/netConnection.h"
#include "editor/editor.h"

#if defined HAVE_SHAPEBASE
#include "game/shapeBase.h"
#include "odeshape.h"
#endif // defined HAVE_SHAPEBASE

#if defined HAVE_FXSCENEOBJECT2D
#include "T2D/fxSceneObject2D.h"
#endif // defined HAVE_FXSCENEOBJECT2D

#include "math/mMathFn.h"
#include "math/mQuat.h"
#include <ode/ode.h>
#include "tbody.h"
#include "tworld.h"
#include "tmass.h"
#include "odehelper.h"



IMPLEMENT_CO_DATABLOCK_V1(tdBodyData);

tdBodyData::tdBodyData()
{
}

tdBodyData::~tdBodyData()
{
}


void tdBodyData::consoleInit()
{
	Parent::consoleInit();
}

void tdBodyData::initPersistFields()
{
	Parent::initPersistFields();
}

bool tdBodyData::preload(bool server, char errorBuffer[256])
{
	return Parent::preload(server, errorBuffer);
}

void tdBodyData::packData(BitStream* stream)
{
	Parent::packData(stream);
}

void tdBodyData::unpackData(BitStream* stream)
{
	Parent::unpackData(stream);
}


IMPLEMENT_CO_NETOBJECT_V1(tdBody);


tdBody::tdBody() {
	// mNetFlags.set(ScopeAlways | Ghostable);
	mNetFlags.clear(Ghostable);
	worldsim = 0;
	body = 0;
	iscreated = 0;
#if defined HAVE_SHAPEBASE
	shapeid = 0;
#endif // defined HAVE_SHAPEBASE

#if defined HAVE_FXSCENEOBJECT2D
	sceneobjectid = 0;
#endif // defined HAVE_FXSCENEOBJECT2D
}

tdBody::~tdBody() {
}

bool tdBody::onAdd() {
	if(!Parent::onAdd()) {
		return false;
	}
	
	tdWorld *w;
	if(NULL == (w = (tdWorld *)Sim::findObject(worldsim))) {
		if(isGhost()) {
			w = (tdWorld *)Sim::findObject(Con::getIntVariable("$odeworldclient"));
		} else {
			w = (tdWorld *)Sim::findObject(Con::getIntVariable("$odeworld"));
		}
	}
	
	if(NULL == w) {
		iscreated = 0;
	} else {
		body = dBodyCreate(w->world);
		iscreated = 1;
	}
	
	return true;
}

void tdBody::onRemove() {
	Parent::onRemove();
	if(iscreated) {
		dBodyDestroy(body);
		iscreated = 0;
	}
}

void tdBody::consoleInit()
{
	Parent::consoleInit();
}

bool tdBody::onNewDataBlock(GameBaseData* dptr)
{
	mDataBlock = dynamic_cast<tdBodyData*>(dptr);
	if (!mDataBlock || !Parent::onNewDataBlock(dptr))
		return false;

   scriptOnNewDataBlock();
   return true;
}

void tdBody::initPersistFields()
{
	Parent::initPersistFields();
	
	addField("worldsim", TypeS32,
		Offset(worldsim, tdBody), "The tdWorld the body is in");
#if defined HAVE_SHAPEBASE
	addField("shapeid", TypeS32,
		Offset(shapeid, tdBody), "The ShapeBase-derived object the body represents");
#endif // defined SHAPEBASE
#if defined HAVE_FXSCENEOBJECT2D
	addField("sceneobjectid", TypeS32,
		Offset(sceneobjectid, tdBody), "The fxSceneObject2D-derived object the body represents");
#endif // defined HAVE_FXSCENEOBJECT2D

}

void tdBody::processTick(const Move* move)
{
	Parent::processTick(move);

	if(0==Con::getBoolVariable("$pref::ODEPhysics::Client::EnableODE",1) && isGhost()) {
		return;
	}

	if(0 == iscreated) {
		return;
	}
	
	float dampingfactor = Con::getFloatVariable("$pref::ODEPhysics::Server::DampingFactor", 0);
	if(0.0f != dampingfactor) {
		damp(dampingfactor);
	}

#if defined HAVE_SHAPEBASE
	if(0 < shapeid) {
		if(gEditingMission ||  // Global Variables make Baby Jesus Cry
		   (isGhost() && 0==Con::getIntVariable("$pref::ODEPhysics::Client::EnableODE", 1))) {
			// UpdateFromShape();
		} else {
			UpdateShape();
		}
	}
#endif // defined HAVE_SHAPEBASE

/* #if defined HAVE_FXSCENEOBJECT2D
	UpdateSceneObject();
#endif // defined HAVE_FXSCENEOBJECT2D */
}

void tdBody::interpolateTick(F32 delta)
{
	Parent::interpolateTick(delta);
}

void tdBody::advanceTime(F32 dt)
{
	Parent::advanceTime(dt);
}

U32 tdBody::packUpdate(NetConnection *conn, U32 mask, BitStream *stream)
{
	U32 ret = Parent::packUpdate(conn, mask, stream);
	if(stream->writeFlag(mask&NewWorld && 0<worldsim)) {
		S32 cid = conn->getGhostIndex((NetObject *)Sim::findObject(worldsim));
		stream->writeInt(cid,32);
	}
	if(stream->writeFlag(mask&NewShape && 0<shapeid)) {
		S32 cid = conn->getGhostIndex((NetObject *)Sim::findObject(shapeid));
		stream->writeInt(cid,32);
	}
	
	return ret;
}

void tdBody::unpackUpdate(NetConnection *conn, BitStream *stream)
{
	Parent::unpackUpdate(conn, stream);
	S32 i;
	NetObject *t;
	
	// World
	if(stream->readFlag()) {
		i = stream->readInt(32);
		t = conn->resolveGhost(i);
		if(NULL != t) worldsim = t->getId();
		if(0 == worldsim) {
			worldsim = Con::getIntVariable("$odeworldclient", 0);
		}
		tdWorld *w;
		w = (tdWorld *)Sim::findObject(worldsim);
		if(NULL == w) {
			iscreated = 0;
		} else {
			body = dBodyCreate(w->world);
			iscreated = 1;
		}
	}
	
	// Shape
	if(stream->readFlag()) {
		i = stream->readInt(32);
		t = conn->resolveGhost(i);
		if(NULL != t) shapeid = t->getId();

		ODEShape *s;
		s = (ODEShape *)Sim::findObject(shapeid);
		if(NULL != s) {
			s->odeobj = getId();
		}
	}
	
}

void tdBody::damp(float factor) {
	const dReal *v = NULL;
	v = dBodyGetLinearVel(body);
	dReal x,y,z;
	if(NULL == v) return;
	x = v[0];
	y = v[1];
	z = v[2];
	dBodyAddForce(body, v[0]*factor, v[1]*factor, v[2]*factor);
	
	v = dBodyGetAngularVel(body);
	dBodyAddTorque(body, v[0]*factor, v[1]*factor, v[2]*factor);
}

void tdBody::SetTransform(const char *line) {
	float x,y,z;
	F32 anglex, angley, anglez, anglew;
	
	dSscanf(line, "%f %f %f %f %f %f %f",
			&x, &y, &z,
			&anglex, &angley, &anglez, &anglew);
	
	dBodySetPosition(body, x, y, z);
	AngAxisF foo(Point3F(anglex, angley, anglez), anglew);
	QuatF q(foo);
	dQuaternion r;
	
	r[0] = -q.w;
	r[1] = q.x;
	r[2] = q.y;
	r[3] = q.z;
	dBodySetQuaternion(body, r);
}

#if defined HAVE_SHAPEBASE
void tdBody::UpdateFromShape() {

	if(0 == iscreated) return;

	ShapeBase *s;
	s = (ShapeBase *)Sim::findObject(shapeid);
	if(NULL == s) return;
	
	dBodySetLinearVel(body,0,0,0);
	
	
	const MatrixF& tmat = s->getTransform();
	
	Point3F pos; // Position of ShapeBase
	AngAxisF aa(tmat); // AngAxis from ShapeBase
	QuatF q(aa); // Quat of shapeBase
	dQuaternion odeq;
	
	tmat.getColumn(3,&pos);
	dBodySetPosition(body, pos.x, pos.y, pos.z);
	
	odeq[0] = -q.w;
	odeq[1] = q.x;
	odeq[2] = q.y;
	odeq[3] = q.z;
	dBodySetQuaternion(body, odeq);

}

void tdBody::UpdateShape() {

	if(0 == iscreated) return;

	const dReal *v, *p, *r; // Velocity, Position & Rotation of ODE body

	ShapeBase *s;
	ODEShape *odeshape;
	Sim::findObject(shapeid,s);
	if(NULL == s) return;

	// Velocity for linear predictions
	v = dBodyGetLinearVel(body);
	s->setVelocity(Point3F(v[0],v[1],v[2]));
	v = dBodyGetAngularVel(body);
	odeshape = dynamic_cast<ODEShape *>(s);
	if(NULL != odeshape) {
		odeshape->setRotVelocity(Point3F(v[0],v[1],v[2]));
	}

	// Update ShapeBase with position and rotation
	p = dBodyGetPosition(body);
	r = dBodyGetQuaternion(body);

	// take the ODE quat and turn into an AngAxisF
	QuatF odeq(r[1], r[2], r[3], -r[0]);
	AngAxisF aa(odeq);
	MatrixF mat = s->getTransform();
	odeq.setMatrix(&mat);
	Point3F pos(p[0], p[1], p[2]); // Position of the object	
	mat.setPosition(pos);

	s->setTransform(mat);
}

void tdBody::UpdateNode() {
}

void tdBody::UpdateFromNode() {
}

#endif // defined HAVE_SHAPEBASE


#if defined HAVE_FXSCENEOBJECT2D
void tdBody::UpdateSceneObject() {

	const dReal *v, *p, *r; // Velocity, Position & Rotation of ODE body

	fxSceneObject2D *s;
	s = (fxSceneObject2D *)Sim::findObject(sceneobjectid);
	if(NULL == s) return;
	
	v = dBodyGetLinearVel(body);
	s->setLinearVelocity(fxVector2D(v[0],v[1]));
	
	v = dBodyGetAngularVel(body);
	s->setAngularVelocity(v[2]);
	
	p = dBodyGetPosition(body);
	s->setPosition(fxVector2D(v[0],v[1]));
	return;
	r = dBodyGetQuaternion(body);
	
	
	// See GetEulerAngles above for more help on this one
	
	const dReal *mat;
	F32 angle_y, angle_x, angle_z;
	F32 trxx, tryy, C, D; // Temp variables

	mat = dBodyGetRotation(body);

	
	angle_y = D =  mAsin( mat[2]);		/* Calculate Y-axis angle */
	C		   =  mCos( angle_y );
	
	if ( fabs( C ) > 0.005 ) {			 /* Gimball lock? */
		trxx	  =  mat[10] / C;		   /* No, so get X-axis angle */
		tryy	  = -mat[6]  / C;
		angle_x  = mAtan( tryy, trxx );
		trxx	  =  mat[0] / C;			/* Get Z-axis angle */
		tryy	  = -mat[1] / C;
		angle_z  = mAtan( tryy, trxx );
	} else {								/* Gimball lock has occurred */
		angle_x  = 0;						 /* Set X-axis angle to zero */
		trxx	  =  mat[5];				 /* And calculate Z-axis angle */
		tryy	  =  mat[4];
		angle_z  = mAtan( tryy, trxx );
	}
	
	if (angle_z < 0) angle_z += 2 * M_PI;
	// Need to subtract 90 in here?
	s->setRotation(mRadToDeg(angle_z));
}
#endif // defined HAVE_FXSCENEOBJECT2D
