#include "odeshape.h"
#include "torque_ver.h"
#include "core/bitStream.h"
#include "math/mathIO.h"
#include "ode/ode.h"
#include "tbody.h"
#include "tworld.h"
#include "tjoint.h"
#include "editor/editor.h"

#if defined HAVE_SHAPEBASE

const U32 sClientCollisionContactMask = (TerrainObjectType     | InteriorObjectType |
                                         StaticShapeObjectType | VehicleObjectType  |
                                         PlayerObjectType      | StaticTSObjectType |
										ProjectileObjectType);

const U32 sServerCollisionContactMask = (sClientCollisionContactMask);

IMPLEMENT_CO_DATABLOCK_V1(ODEShapeData);

ODEShapeData::ODEShapeData()
{
}

ODEShapeData::~ODEShapeData()
{

}

void ODEShapeData::consoleInit()
{
   Parent::consoleInit();
}

void ODEShapeData::initPersistFields()
{
   Parent::initPersistFields();
}

bool ODEShapeData::preload(bool server, char errorBuffer[256])
{
   return Parent::preload(server, errorBuffer);
}

void ODEShapeData::packData(BitStream* stream)
{
   Parent::packData(stream);
}

void ODEShapeData::unpackData(BitStream* stream)
{
   Parent::unpackData(stream);
}

IMPLEMENT_CO_NETOBJECT_V1(ODEShape);

ODEShape::ODEShape()
{
	contactgroup = 0;
	worldsim = 0;
	odeobj = 0;
	disablecollisions = false;
	mTypeMask |= StaticShapeObjectType;
}

ODEShape::~ODEShape()
{

}

void ODEShape::initPersistFields()
{
   Parent::initPersistFields();
   
	addField("contactgroup", TypeS32,
		Offset(contactgroup, ODEShape), "The joint group to add contact joints to");
	addField("worldsim", TypeS32,
		Offset(worldsim, ODEShape), "The tdWorld the body is in");
	addField("odeobj", TypeS32,
		Offset(odeobj, ODEShape), "The ode body this represents");
	addField("disablecollisions", TypeS32,
		Offset(disablecollisions, ODEShape), "Disable collision detection for this object");

}

void ODEShape::consoleInit()
{
   Parent::consoleInit();
}

bool ODEShape::onAdd()
{
   if(!Parent::onAdd() || !mDataBlock)
      return false;

   mTypeMask |= ShapeBaseObjectType;

   // Create a new convex.
   AssertFatal(mDataBlock->collisionDetails[0] != -1, "Error, a rigid shape must have a collision-1 detail!");
   mConvex.mObject    = this;
   mConvex.pShapeBase = this;
   mConvex.hullId     = 0;
   mConvex.box        = mObjBox;
   mConvex.box.min.convolve(mObjScale);
   mConvex.box.max.convolve(mObjScale);
   mConvex.findNodeTransform();

   addToScene();

   if (isServerObject())
   {
      scriptOnAdd();
   }
   
   if(isGhost()) { Con::evaluatef("commandToServer('SetupODEShape', %i);", getNetIndex()); }

	if(isGhost()) {
		if(Con::getIntVariable("$odecollisionjgclient", -1) > 0) {
			contactgroup=Con::getIntVariable("$odecollisionjgclient");
		}
	}


   return true;
}

void ODEShape::onRemove()
{
   scriptOnRemove();
   removeFromScene();
   Parent::onRemove();
}


bool ODEShape::onNewDataBlock(GameBaseData* dptr)
{
   mDataBlock = dynamic_cast<ODEShapeData*>(dptr);
   if (!mDataBlock || !Parent::onNewDataBlock(dptr))
      return false;

   scriptOnNewDataBlock();
   return true;
}

void ODEShape::processTick(const Move *move)
{
   Parent::processTick(move);
   
   if(0 == Con::getBoolVariable("$pref::ODEPhysics::Client::EnableODE",1) && isGhost()) {
      return;
   }
   
   Point3F pos = Parent::getPosition();
   
   // if(20 > mVelocity.len() && 0 == gEditingMission) {
   // Sanity check for shortly after the object is created before the physics properly kicks in
   //   pos += TickMs * mVelocity/1000;
   // }
   
	if(isGhost()) {
		char varname[12];
		if(worldsim == 0) {
			dSprintf(varname, sizeof(varname)-1, "%i.worldsim", getId());
			worldsim = Con::getIntVariable(varname);
			// if(0 != worldsim) Con::printf("odeshape got a new worldsim! %i", worldsim);
		}
		if(odeobj == 0) {
			dSprintf(varname, sizeof(varname)-1, "%i.odeobj", getId());
			odeobj = Con::getIntVariable(varname);
			// if(0 != odeobj) Con::printf("odeshape got a new odeobj! %i", odeobj);
		}
	}
	
	if(worldsim == 0 || contactgroup == 0 || odeobj == 0) {
 		return; // Can't do anything sensible without all three
	}
	
	//if(isGhost()) {
	//	((tdBody *)Sim::findObject(odeobj))->UpdateShape();
	//}

   MatrixF mat = getTransform();
   mat.setPosition(pos);
   Parent::setTransform(mat);
   Parent::setRenderTransform(mat);
   setMaskBits(PositionMask);


	if(1 == Con::getIntVariable("$pref::ODEPhysics::Server::DisableCollision",0) || disablecollisions) {
		return;
	}
   // Update collisions
   updateWorkingCollisionSet();

   // Test collision
   float tolerance = Con::getFloatVariable("$pref::ODEPhysics::Server::Tolerance",1);
   CollisionList mCollisionList;
   mCollisionList.count = 0;
   
   mConvex.transform = &mat;
   MatrixF cmat = mConvex.getTransform();

   CollisionState *state = mConvex.findClosestState(cmat, getScale(), tolerance);
   if(state && state->dist <= tolerance)
   {
      mConvex.getCollisionInfo(cmat, getScale(), &mCollisionList, tolerance);
   }

	int i;
	for(i=0;i<mCollisionList.count;i++) {
		// Create contact joints if necessary
		if(0 != Con::getIntVariable("$pref::ODEPhysics::Server::AutoContacts", 1) &&
			 0 == gEditingMission) {
			 // Don't create contacts if we're in an editor
		
			dContact contact;
			
			contact.surface.mode = 0;
			contact.surface.mu =
				Con::getFloatVariable("$pref::ODEPhysics::Server::ContactMu");
			contact.surface.mu2 = 0;
			contact.surface.bounce = 0;
			contact.surface.bounce_vel = 0;
			contact.surface.soft_erp = 0;
			contact.surface.soft_cfm = 0;
			contact.surface.motion1 = 0;
			contact.surface.motion2 = 0;
			contact.surface.slip1 = 0;
			contact.surface.slip2 = 0;
			
			if(0 != Con::getIntVariable("$pref::ODEPhysics::Server::SoftContact", 0)) {
				contact.surface.mode |= dContactSoftERP;
				contact.surface.mode |= dContactSoftCFM;
				contact.surface.soft_erp =
					Con::getFloatVariable("$pref::ODEPhysics::Server::ContactERP");
				contact.surface.soft_cfm =
					Con::getFloatVariable("$pref::ODEPhysics::Server::ContactCFM");
			}
			if(0 < Con::getFloatVariable("$pref::ODEPhysics::Server::ContactBounce")) {
				contact.surface.mode |= dContactBounce;
				contact.surface.bounce =
					Con::getFloatVariable("$pref::ODEPhysics::Server::ContactBounce");
			}
			
			contact.geom.depth = Con::getFloatVariable("$pref::ODEPhysics::Server::NormalMultipler", 1) * 
				(mCollisionList.collision[i].distance-tolerance);
			contact.geom.pos[0] = mCollisionList.collision[i].point.x;
			contact.geom.pos[1] = mCollisionList.collision[i].point.y;
			contact.geom.pos[2] = mCollisionList.collision[i].point.z;
			contact.geom.normal[0] = mCollisionList.collision[i].normal.x;
			contact.geom.normal[1] = mCollisionList.collision[i].normal.y;
			contact.geom.normal[2] = mCollisionList.collision[i].normal.z;
			contact.geom.g1 = 0;
			contact.geom.g2 = 0;
			
			// Check the other object to see if it's an odeobj
			char varname[12];
			S32 otherodeobj;
			dSprintf(varname, sizeof(varname)-1, "%i.odeobj", mCollisionList.collision[i].object->getId());
			otherodeobj = Con::getIntVariable(varname,0);
			
			dJointID joint;
			dWorldID w = ((tdWorld *)Sim::findObject(worldsim))->world;
			
			dJointGroupID g = ((tdJointGroup *)Sim::findObject(contactgroup))->group;
			joint = dJointCreateContact(w,g,&contact);
			dJointAttach(joint, ((tdBody *)Sim::findObject(odeobj))->body,
				(0 == otherodeobj || otherodeobj == odeobj)?0:((tdBody *)Sim::findObject(otherodeobj))->body);
		}
	
		// The script callback
	
		// ODEShapeData::onCollision(%this, %obj, %col, %vec, %len)
		char *normalArg = Con::getArgBuffer(64);
		float depth = Con::getFloatVariable("$pref::ODEPhysics::Server::NormalMultipler", 1) * 
				(mCollisionList.collision[i].distance-tolerance);
		dSprintf(normalArg, 64, "%f %f %f",
			mCollisionList.collision[i].normal.x,
			mCollisionList.collision[i].normal.y, 
			mCollisionList.collision[i].normal.z);
		Con::executef(mDataBlock, 5, "onCollision",
			scriptThis(),
			Con::getIntArg(mCollisionList.collision[i].object->getId()),
			normalArg,
			Con::getFloatArg(depth));
	}
}

void ODEShape::interpolateTick(F32 delta)
{
   Parent::interpolateTick(delta);
   
   if(0 == Con::getBoolVariable("$pref::ODEPhysics::Client::Interpolation", 1)) {
	return;
   }
   
   if(0.0f == delta) {
	return;
   }
  Point3F pos = Parent::getPosition();
   
   if(20 > mVelocity.len() && 0 == gEditingMission) {
// Sanity check for shortly after the object is created before the physics properly kicks in
      pos -= delta * mVelocity;
   }
   MatrixF mat = getTransform();
   mat.setPosition(pos);
   Parent::setTransform(mat);
   Parent::setRenderTransform(mat);
}

void ODEShape::advanceTime(F32 dt)
{
   Parent::advanceTime(dt);
}

void ODEShape::setVelocity(const VectorF& v) {
	mVelocity = v;
}

void ODEShape::updateWorkingCollisionSet()
{
   Box3F convexBox = mConvex.getBoundingBox(getTransform(), getScale());
   //F32 len = (mRigid.linVelocity.len() + 50) * TickSec;
   // HACK: just guessing at a possible bounding box
   F32 len = 100 * TickSec;
   F32 l = (len * 1.1) + 0.1;  // fudge factor
   convexBox.min -= Point3F(l, l, l);
   convexBox.max += Point3F(l, l, l);

   disableCollision();
   mConvex.updateWorkingList(convexBox, isGhost() ? sClientCollisionContactMask : sServerCollisionContactMask);
   enableCollision();
}

U32 ODEShape::packUpdate(NetConnection *conn, U32 mask, BitStream *stream)
{
   U32 retMask = Parent::packUpdate(conn, mask, stream);
   if (stream->writeFlag(mask & PositionMask))
   {
      stream->writeAffineTransform(mObjToWorld);
	  mathWrite(*stream, mVelocity); 
   }

   return retMask;
}
void ODEShape::unpackUpdate(NetConnection *conn, BitStream *stream)
{
   Parent::unpackUpdate(conn, stream);
   if (stream->readFlag())
   {
      MatrixF mat;
      stream->readAffineTransform(&mat);
      Parent::setTransform(mat);
      Parent::setRenderTransform(mat);
	  
	  VectorF vel;
	  mathRead(*stream, &vel);
	  mVelocity = vel;
	  
   }
   if(isClientObject() && 0 != odeobj) {
      tdBody *b = (tdBody *)Sim::findObject(odeobj);
	  b->UpdateFromShape();
   }
}

#endif // defined HAVE_SHAPEBASE
