#include "odeshape.h"
#include "torque_ver.h"
#include "core/bitStream.h"
#include "math/mathIO.h"
#include "ode/ode.h"
#include "tbody.h"
#include "tworld.h"
#include "tjoint.h"
#include "editor/editor.h"

#if defined HAVE_SHAPEBASE

const U32 sClientCollisionContactMask = (TerrainObjectType     | InteriorObjectType |
                                         StaticShapeObjectType | VehicleObjectType  |
                                         PlayerObjectType      | StaticTSObjectType);

const U32 sServerCollisionContactMask = (sClientCollisionContactMask);

IMPLEMENT_CO_DATABLOCK_V1(ODEShapeData);

ODEShapeData::ODEShapeData()
{
}

ODEShapeData::~ODEShapeData()
{

}

void ODEShapeData::consoleInit()
{
   Parent::consoleInit();
}

void ODEShapeData::initPersistFields()
{
   Parent::initPersistFields();
}

bool ODEShapeData::preload(bool server, char errorBuffer[256])
{
   return Parent::preload(server, errorBuffer);
}

void ODEShapeData::packData(BitStream* stream)
{
   Parent::packData(stream);
}

void ODEShapeData::unpackData(BitStream* stream)
{
   Parent::unpackData(stream);
}

IMPLEMENT_CO_NETOBJECT_V1(ODEShape);

ODEShape::ODEShape()
{
	contactgroup = 0;
	worldsim = 0;
	odeobj = 0;
	mTypeMask |= StaticShapeObjectType;
}

ODEShape::~ODEShape()
{

}

void ODEShape::initPersistFields()
{
   Parent::initPersistFields();
   
	addField("contactgroup", TypeS32,
		Offset(contactgroup, ODEShape), "The joint group to add contact joints to");
	addField("worldsim", TypeS32,
		Offset(worldsim, ODEShape), "The tdWorld the body is in");
	addField("odeobj", TypeS32,
		Offset(odeobj, ODEShape), "The ode body this represents");


}

void ODEShape::consoleInit()
{
   Parent::consoleInit();
}

bool ODEShape::onAdd()
{
   vel.x = 0.1f;
   time = 0;

   if(!Parent::onAdd() || !mDataBlock)
      return false;

   mTypeMask |= ShapeBaseObjectType;

   // Create a new convex.
   AssertFatal(mDataBlock->collisionDetails[0] != -1, "Error, a rigid shape must have a collision-1 detail!");
   mConvex.mObject    = this;
   mConvex.pShapeBase = this;
   mConvex.hullId     = 0;
   mConvex.box        = mObjBox;
   mConvex.box.min.convolve(mObjScale);
   mConvex.box.max.convolve(mObjScale);
   mConvex.findNodeTransform();

   addToScene();

   if (isServerObject())
   {
      scriptOnAdd();
   }

   return true;
}

void ODEShape::onRemove()
{
   scriptOnRemove();
   removeFromScene();
   Parent::onRemove();
}


bool ODEShape::onNewDataBlock(GameBaseData* dptr)
{
   mDataBlock = dynamic_cast<ODEShapeData*>(dptr);
   if (!mDataBlock || !Parent::onNewDataBlock(dptr))
      return false;

   scriptOnNewDataBlock();
   return true;
}

void ODEShape::processTick(const Move *move)
{
   Parent::processTick(move);

   // Move the object back and forth
   Point3F pos = Parent::getPosition();
	
   MatrixF mat = getTransform();
   mat.setPosition(pos);
   Parent::setTransform(mat);
   Parent::setRenderTransform(mat);
   setMaskBits(PositionMask);
	
   // Update collisions
   updateWorkingCollisionSet();

   // Test collision
   float tolerance = 1.0f;
   CollisionList mCollisionList;
   mCollisionList.count = 0;
   
   mConvex.transform = &mat;
   MatrixF cmat = mConvex.getTransform();

   CollisionState *state = mConvex.findClosestState(cmat, getScale(), tolerance);
   if(state && state->dist <= tolerance)
   {
      mConvex.getCollisionInfo(cmat, getScale(), &mCollisionList, tolerance);
   }

	int i;
	for(i=0;i<mCollisionList.count;i++) {
	
		// Create contact joints if necessary
		if(0 != Con::getIntVariable("$pref::ODEPhysics::Server::AutoContacts") &&
			 mCollisionList.collision[i].distance > 0 &&
			 0 == gEditingMission) { // Don't create contacts if we're in an editor
		
			if(worldsim == 0 || contactgroup == 0 || odeobj == 0) {
				continue; // Can't do anything sensible without all three
			}
			
			dContact contact;
			
			contact.surface.mode = 0;
			contact.surface.mu = 500;
			contact.surface.mu2 = 0;
			contact.surface.bounce =
				Con::getFloatVariable("$pref::ODEPhysics::Server::ContactBounce");
			contact.surface.bounce_vel = 0;
			contact.surface.soft_erp = 
				Con::getFloatVariable("$pref::ODEPhysics::Server::ContactERP");
			contact.surface.soft_cfm =
				Con::getFloatVariable("$pref::ODEPhysics::Server::ContactCFM");
			contact.surface.motion1 = 0;
			contact.surface.motion2 = 0;
			contact.surface.slip1 = 0;
			contact.surface.slip2 = 0;
			
			if(0 != Con::getIntVariable("$pref::ODEPhysics::Server::SoftContact")) {
				contact.surface.mode |= dContactSoftERP;
				contact.surface.mode |= dContactSoftCFM;
			}
			contact.surface.mode |= dContactBounce;
			
			contact.geom.depth = mCollisionList.collision[i].distance/5;
			contact.geom.pos[0] = mCollisionList.collision[i].point.x;
			contact.geom.pos[1] = mCollisionList.collision[i].point.y;
			contact.geom.pos[2] = mCollisionList.collision[i].point.z;
			contact.geom.normal[0] = mCollisionList.collision[i].normal.x;
			contact.geom.normal[1] = mCollisionList.collision[i].normal.y;
			contact.geom.normal[2] = mCollisionList.collision[i].normal.z;
			contact.geom.g1 = 0;
			contact.geom.g2 = 0;
			
			dJointID joint;
			dWorldID w = ((tdWorld *)Sim::findObject(worldsim))->world;
			dJointGroupID g = ((tdJointGroup *)Sim::findObject(contactgroup))->group;
			joint = dJointCreateContact(w,g,&contact);
			dJointAttach(joint, ((tdBody *)Sim::findObject(odeobj))->body, 0);
		}
	
		// The script callback
	
		// ODEShapeData::onCollision(%this, %obj, %col, %vec, %len)
		char *normalArg = Con::getArgBuffer(64);
		dSprintf(normalArg, 64, "%f %f %f",
			mCollisionList.collision[i].normal.x,
			mCollisionList.collision[i].normal.y, 
			mCollisionList.collision[i].normal.z);
		Con::executef(mDataBlock, 5, "onCollision",
			scriptThis(),
			Con::getIntArg(mCollisionList.collision[i].object->getId()),
			normalArg,
			Con::getFloatArg(mCollisionList.collision[i].distance));
	}


}

void ODEShape::interpolateTick(F32 delta)
{
   Parent::interpolateTick(delta);
}

void ODEShape::advanceTime(F32 dt)
{
   Parent::advanceTime(dt);
}

void ODEShape::updateWorkingCollisionSet()
{
   Box3F convexBox = mConvex.getBoundingBox(getTransform(), getScale());
   //F32 len = (mRigid.linVelocity.len() + 50) * TickSec;
   // HACK: just guessing at a possible bounding box
   F32 len = 100 * TickSec;
   F32 l = (len * 1.1) + 0.1;  // fudge factor
   convexBox.min -= Point3F(l, l, l);
   convexBox.max += Point3F(l, l, l);

   disableCollision();
   mConvex.updateWorkingList(convexBox, isGhost() ? sClientCollisionContactMask : sServerCollisionContactMask);
   enableCollision();
}

U32 ODEShape::packUpdate(NetConnection *conn, U32 mask, BitStream *stream)
{
   U32 retMask = Parent::packUpdate(conn, mask, stream);
   if (stream->writeFlag(mask & PositionMask))
   {
      stream->writeAffineTransform(mObjToWorld);
      mathWrite(*stream, mObjScale);
   }

   return retMask;
}
void ODEShape::unpackUpdate(NetConnection *conn, BitStream *stream)
{
   Parent::unpackUpdate(conn, stream);
   if (stream->readFlag())
   {
      MatrixF mat;
      stream->readAffineTransform(&mat);
      Parent::setTransform(mat);
      Parent::setRenderTransform(mat);

      VectorF scale;
      mathRead(*stream, &scale);
      setScale(scale);
   }
}

#endif // defined HAVE_SHAPEBASE
