#include "console/simBase.h"
#include "console/consoleObject.h"
#include "console/consoleTypes.h"
#include "math/mMathFn.h"
#include <ode/ode.h>
#include "tbody.h"
#include "tworld.h"
#include "tmass.h"
#include "odehelper.h"

tdBody::tdBody() {
	worldsim = 0;
}

bool tdBody::onAdd() {
	if(!Parent::onAdd()) {
		return false;
	}

	tdWorld *w;
	w = (tdWorld *)Sim::findObject(worldsim);

	if(NULL == w) {
		iscreated = 0;
	} else {
		body = dBodyCreate(w->world);
		iscreated = 1;
	}

	return true;
}

void tdBody::onRemove() {
	if(iscreated) {
		dBodyDestroy(body);
		iscreated = 0;
	}
	Parent::onRemove();
}

IMPLEMENT_CONOBJECT(tdBody);												   

void tdBody::initPersistFields()
{
	Parent::initPersistFields();

	addField("worldsim", TypeS32,
		Offset(worldsim, tdBody), "The tdWorld the body is in");
}

ConsoleMethod(tdBody, Create, bool, 3, 3,
	"(world) "
	"Create Body in World") {

	argc; argv;
	tdWorld *w;
	w = (tdWorld *)Sim::findObject(dAtoi(argv[2]));
	if(NULL == w) {
		return 0;
	}

	object->body = dBodyCreate(w->world);
	object->iscreated = 1;
	return 1;
}

ConsoleMethod(tdBody, SetPosition, void, 3, 5,
	"(x,y,z) or (\"x y z\") "
	"Set Body's Position") {

	argc; argv;
	float x, y, z;
	if(0 != ODEOneOrThree(argc,argv,&x,&y,&z)) {
		Con::errorf("Passed incorrect arguments to %s::%s", object->getClassName(), argv[0]);
	} else {
		dBodySetPosition(object->body, x, y, z);
	}
}

ConsoleMethod(tdBody, GetPosition, const char *, 2, 2,
	"() "
	"Get Body's Position") {

	char *ret = Con::getReturnBuffer(256);
	const dReal *p;

	p = dBodyGetPosition(object->body);

	dSprintf(ret, 256, "%f %f %f", p[0], p[1], p[2]);
	return ret;
}

ConsoleMethod(tdBody, SetLinearVel, void, 3, 5,
	"(x,y,z) or (\"x y z\") "
	"Set Body's Linear Velocity") {

	argc; argv;
	float x, y, z;
	if(0 != ODEOneOrThree(argc,argv,&x,&y,&z)) {
		Con::errorf("Passed incorrect arguments to %s::%s", object->getClassName(), argv[0]);
	} else {
		dBodySetLinearVel(object->body, x, y, z);
	}
}

ConsoleMethod(tdBody, GetLinearVel, const char *, 2, 2,
	"() "
	"Get Body's Linear Velocity") {

	char *ret = Con::getReturnBuffer(256);
	const dReal *p;

	p = dBodyGetLinearVel(object->body);

	dSprintf(ret, 256, "%f %f %f", p[0], p[1], p[2]);
	return ret;
}

ConsoleMethod(tdBody, SetAngularVel, void, 3, 5,
	"(x,y,z) or (\"x y z\") "
	"Set Body's Angular Velocity") {

	argc; argv;
	float x, y, z;
	if(0 != ODEOneOrThree(argc,argv,&x,&y,&z)) {
		Con::errorf("Passed incorrect arguments to %s::%s", object->getClassName(), argv[0]);
	} else {
		dBodySetAngularVel(object->body, x, y, z);
	}
}

ConsoleMethod(tdBody, GetAngularVel, const char *, 2, 2,
	"() "
	"Get Body's Angular Velocity") {

	char *ret = Con::getReturnBuffer(256);
	const dReal *p;

	p = dBodyGetAngularVel(object->body);

	dSprintf(ret, 256, "%f %f %f", p[0], p[1], p[2]);
	return ret;
}

ConsoleMethod(tdBody, AddForce, void, 3, 5,
	"(fx,fy,fz) or (\"fx fy fz\") "
	"Add a Force to a Body") {

	argc; argv;
	float x, y, z;
	if(0 != ODEOneOrThree(argc,argv,&x,&y,&z)) {
		Con::errorf("Passed incorrect arguments to %s::%s", object->getClassName(), argv[0]);
	} else {
		dBodyAddForce(object->body, x, y, z);
	}
}

ConsoleMethod(tdBody, AddTorque, void, 3, 5,
	"(fx,fy,fz) or (\"fx fy fz\") "
	"Add a Torque to a Body") {

	argc; argv;
	float x, y, z;
	if(0 != ODEOneOrThree(argc,argv,&x,&y,&z)) {
		Con::errorf("Passed incorrect arguments to %s::%s", object->getClassName(), argv[0]);
	} else {
		dBodyAddTorque(object->body, x, y, z);
	}
}

ConsoleMethod(tdBody, AddRelForce, void, 3, 5,
	"(fx,fy,fz) or (\"fx fy fz\") "
	"Add a Relative Force to a Body") {

	argc; argv;
	float x, y, z;
	if(0 != ODEOneOrThree(argc,argv,&x,&y,&z)) {
		Con::errorf("Passed incorrect arguments to %s::%s", object->getClassName(), argv[0]);
	} else {
		dBodyAddRelForce(object->body, x, y, z);
	}
}

ConsoleMethod(tdBody, AddRelTorque, void, 3, 5,
	"(fx,fy,fz) or (\"fx fy fz\") "
	"Add a Relative Torque to a Body") {

	argc; argv;
	
	float x, y, z;
	if(0 != ODEOneOrThree(argc,argv,&x,&y,&z)) {
		Con::errorf("Passed incorrect arguments to %s::%s", object->getClassName(), argv[0]);
	} else {
		dBodyAddRelTorque(object->body, x, y, z);
	}
}

ConsoleMethod(tdBody, AddForceAtPos, void, 4, 8,
	"(fx,fy,fz,px,py,pz) or (\"fx fy fz\",\"px py pz\") "
	"Add a Force to a Body at a specific Position") {

	argc; argv;
	float fx,fy,fz,px,py,pz;
	
	if(0 != ODETwoOrSix(argc,argv,&fx,&fy,&fz,&px,&py,&pz)) {
		Con::errorf("Passed incorrect arguments to %s::%s", object->getClassName(), argv[0]);
	} else {
		dBodyAddForceAtPos(object->body, fx, fy, fz, px, py, pz);
	}
}

ConsoleMethod(tdBody, AddForceAtRelPos, void, 4, 8,
	"(fx,fy,fz,px,py,pz) or (\"fx fy fz\",\"px py pz\") "
	"Add a Force to a Body at a Relative Position") {

	argc; argv;
	float fx,fy,fz,px,py,pz;
	
	if(0 != ODETwoOrSix(argc,argv,&fx,&fy,&fz,&px,&py,&pz)) {
		Con::errorf("Passed incorrect arguments to %s::%s", object->getClassName(), argv[0]);
	} else {
		dBodyAddForceAtRelPos(object->body, fx, fy, fz, px, py, pz);
	}
}

ConsoleMethod(tdBody, AddRelForceAtPos, void, 4, 8,
	"(fx,fy,fz,px,py,pz) or (\"fx fy fz\",\"px py pz\") "
	"Add a Relative Force to a Body at a Position") {

	argc; argv;
	float fx,fy,fz,px,py,pz;
	
	if(0 != ODETwoOrSix(argc,argv,&fx,&fy,&fz,&px,&py,&pz)) {
		Con::errorf("Passed incorrect arguments to %s::%s", object->getClassName(), argv[0]);
	} else {
		dBodyAddRelForceAtPos(object->body, fx, fy, fz, px, py, pz);
	}
}

ConsoleMethod(tdBody, AddRelForceAtRelPos, void, 4, 8,
	"(fx,fy,fz,px,py,pz) or (\"fx fy fz\",\"px py pz\") "
	"Add a Relative Force to a Body at a Relative Position") {

	argc; argv;
	float fx,fy,fz,px,py,pz;
	
	if(0 != ODETwoOrSix(argc,argv,&fx,&fy,&fz,&px,&py,&pz)) {
		Con::errorf("Passed incorrect arguments to %s::%s", object->getClassName(), argv[0]);
	} else {
		dBodyAddRelForceAtRelPos(object->body, fx, fy, fz, px, py, pz);
	}
}

ConsoleMethod(tdBody, SetForce, void, 3, 5,
	"(x,y,z) or (\"x y z\") "
	"Set Body's Force") {

	argc; argv;
	
	float x, y, z;
	if(0 != ODEOneOrThree(argc,argv,&x,&y,&z)) {
		Con::errorf("Passed incorrect arguments to %s::%s", object->getClassName(), argv[0]);
	} else {
		dBodySetForce(object->body, x, y, z);
	}
}

ConsoleMethod(tdBody, GetForce, const char *, 2, 2,
	"() "
	"Get Body's Force") {

	char *ret = Con::getReturnBuffer(256);
	const dReal *p;

	p = dBodyGetForce(object->body);

	dSprintf(ret, 256, "%f %f %f", p[0], p[1], p[2]);
	return ret;
}

ConsoleMethod(tdBody, SetTorque, void, 3, 5,
	"(x,y,z) or (\"x y z\") "
	"Set Body's Torque") {

	argc; argv;
	
	float x, y, z;
	if(0 != ODEOneOrThree(argc,argv,&x,&y,&z)) {
		Con::errorf("Passed incorrect arguments to %s::%s", object->getClassName(), argv[0]);
	} else {
		dBodySetTorque(object->body, x, y, z);
	}
}

ConsoleMethod(tdBody, GetTorque, const char *, 2, 2,
	"() "
	"Get Body's Torque") {

	char *ret = Con::getReturnBuffer(256);
	const dReal *p;

	p = dBodyGetTorque(object->body);

	dSprintf(ret, 256, "%f %f %f", p[0], p[1], p[2]);
	return ret;
}

ConsoleMethod(tdBody, GetRelPointPos, const char *, 3, 5,
	"(px,py,pz) or (\"px py pz\") "
	"Get Body's Relative Position") {

	argc; argv;
	char *ret = Con::getReturnBuffer(256);
	dVector3 p;
	
	float x, y, z;
	if(0 != ODEOneOrThree(argc,argv,&x,&y,&z)) {
		Con::errorf("Passed incorrect arguments to %s::%s", object->getClassName(), argv[0]);
	} else {
		dBodyGetRelPointPos(object->body, x, y, z, p);
		dSprintf(ret, 256, "%f %f %f", p[0], p[1], p[2]);
	}

	return ret;
}

ConsoleMethod(tdBody, GetRelPointVel, const char *, 3, 5,
	"(px,py,pz) or (\"px py pz\") "
	"Get Body's Relative Velocity") {

	argc; argv;
	float x,y,z;
	char *ret = Con::getReturnBuffer(256);
	dVector3 p;

	if(0 != ODEOneOrThree(argc,argv,&x,&y,&z)) {
		Con::errorf("Passed incorrect arguments to %s::%s", object->getClassName(), argv[0]);
	} else {
		dBodyGetRelPointVel(object->body, x, y, z, p);
		dSprintf(ret, 256, "%f %f %f", p[0], p[1], p[2]);
	}
	
	return ret;
}

ConsoleMethod(tdBody, GetPosRelPoint, const char *, 3, 5,
	"(px,py,pz) or (\"px py pz\") "
	"Get Body's Relative Position") {

	argc; argv;
	float x,y,z;
	char *ret = Con::getReturnBuffer(256);
	dVector3 p;

	if(0 != ODEOneOrThree(argc,argv,&x,&y,&z)) {
		Con::errorf("Passed incorrect arguments to %s::%s", object->getClassName(), argv[0]);
	} else {
		dBodyGetPosRelPoint(object->body, x, y, z, p);
		dSprintf(ret, 256, "%f %f %f", p[0], p[1], p[2]);
	}
	
	return ret;
}

ConsoleMethod(tdBody, VectorToWorld, const char *, 3, 5,
	"(px,py,pz) or (\"px py pz\") "
	"Get Body's Vector to World") {

	argc; argv;
	float x,y,z;
	char *ret = Con::getReturnBuffer(256);
	dVector3 p;


	if(0 != ODEOneOrThree(argc,argv,&x,&y,&z)) {
		Con::errorf("Passed incorrect arguments to %s::%s", object->getClassName(), argv[0]);
	} else {
		dBodyVectorToWorld(object->body, x, y, z, p);
		dSprintf(ret, 256, "%f %f %f", p[0], p[1], p[2]);
	}
	
	return ret;
}

ConsoleMethod(tdBody, VectorFromWorld, const char *, 3, 5,
	"(px,py,pz) or (\"px py pz\") "
	"Get Body's Vector from World") {

	argc; argv;
	float x,y,z;
	char *ret = Con::getReturnBuffer(256);
	dVector3 p;

	if(0 != ODEOneOrThree(argc,argv,&x,&y,&z)) {
		Con::errorf("Passed incorrect arguments to %s::%s", object->getClassName(), argv[0]);
	} else {
		dBodyVectorFromWorld(object->body, x, y, z, p);
		dSprintf(ret, 256, "%f %f %f", p[0], p[1], p[2]);
	}
	
	return ret;
}

ConsoleMethod(tdBody, SetFiniteRotationMode, void, 2, 2,
	"(mode) "
	"Set a body's finite rotation mode") {

	argc; argv;
	dBodySetFiniteRotationMode(object->body, dAtoi(argv[2]));
}

ConsoleMethod(tdBody, GetFiniteRotationMode, S32, 2, 2,
	"() "
	"Get a body's finite rotation mode") {

	return dBodyGetFiniteRotationMode(object->body);
}

ConsoleMethod(tdBody, SetFiniteRotationAxis, void, 3, 5,
	"(x,y,z) or (\"x y z\") "
	"Set Body's Finite Rotation Axis") {

	argc; argv;
	float x,y,z;

	if(0 != ODEOneOrThree(argc,argv,&x,&y,&z)) {
		Con::errorf("Passed incorrect arguments to %s::%s", object->getClassName(), argv[0]);
	} else {
		dBodySetFiniteRotationAxis(object->body, dAtof(argv[2]), dAtof(argv[3]),
			dAtof(argv[4]));
	}
}

ConsoleMethod(tdBody, dBodyGetFiniteRotationAxis, const char *, 2, 2,
	"() "
	"Get Body's Finite Rotation Axis") {

	argc; argv;
	char *ret = Con::getReturnBuffer(256);
	dVector3 p;

	dBodyGetFiniteRotationAxis(object->body, p);

	dSprintf(ret, 256, "%f %f %f", p[0], p[1], p[2]);
	return ret;
}

ConsoleMethod(tdBody, Enable, void, 2, 2,
	"() "
	"Enable Body") {

	dBodyEnable(object->body);
}

ConsoleMethod(tdBody, Disable, void, 2, 2,
	"() "
	"Disable Body") {

	dBodyDisable(object->body);
}

ConsoleMethod(tdBody, IsEnabled, S32, 2, 2,
	"() "
	"See if body is Enabled") {

	return dBodyIsEnabled(object->body);
}

ConsoleMethod(tdBody, SetGravityMode, void, 2, 2,
	"(mode) "
	"Set a body's gravity mode") {

	argc; argv;
	dBodySetGravityMode(object->body, dAtoi(argv[2]));
}

ConsoleMethod(tdBody, GetGravityMode, S32, 2, 2,
	"() "
	"Get a body's gravity mode") {

	return dBodyGetGravityMode(object->body);
}

ConsoleMethod(tdBody, GetNumJoints, S32, 2, 2,
	"() "
	"Get a number of joints on a body") {

	return dBodyGetNumJoints(object->body);
}

ConsoleMethod(tdBody, SetMass, void, 3, 3,
	"(mass) "
	"Set a body's mass") {

	argc;argv;

	tdMass *m;
	m = (tdMass *)Sim::findObject(dAtoi(argv[2]));
	dBodySetMass(object->body, &m->mass);

	object->mass = dAtoi(argv[2]);
}

ConsoleMethod(tdBody, GetMass, S32, 2, 2,
	"() "
	"Get a body's mass") {

	return object->mass;
}

ConsoleMethod(tdBody, Damp, void, 2, 3,
	"([factor]) "
	"Damp a body's linear and angular velocity by factor [try -0.02]") {

	const dReal *v;
	dReal factor;

	argc; argv;
	if(argc == 2) {
		factor = -0.02;
	} else {
		factor = dAtof(argv[2]);
	}
	v = dBodyGetLinearVel(object->body);
	dBodyAddForce(object->body, v[0]*factor, v[1]*factor, v[2]*factor);
	
	v = dBodyGetAngularVel(object->body);
	dBodyAddTorque(object->body, v[0]*factor, v[1]*factor, v[2]*factor);
}

ConsoleMethod(tdBody, SetRotation, void, 11, 11,
	"(9 numbers go here) "
	"Set a body's rotation. 3x3 matrix, rows-first") {

	argc; argv;
	dMatrix3 r;
	r[2]=dAtof(argv[2]);
	r[3]=dAtof(argv[3]);
	r[4]=dAtof(argv[4]);
	r[5]=dAtof(argv[5]);
	r[6]=dAtof(argv[6]);
	r[7]=dAtof(argv[7]);
	r[8]=dAtof(argv[8]);
	r[9]=dAtof(argv[9]);
	r[10]=dAtof(argv[10]);
	dBodySetRotation(object->body, r);
}

ConsoleMethod(tdBody, SetQuaternion, void, 14, 14,
	"(12 numbers go here) "
	"Set a body's rotation. 4x3 quaternion, rows-first") {

	argc; argv;
	dQuaternion r;
	r[2]=dAtof(argv[2]);
	r[3]=dAtof(argv[3]);
	r[4]=dAtof(argv[4]);
	r[5]=dAtof(argv[5]);
	r[6]=dAtof(argv[6]);
	r[7]=dAtof(argv[7]);
	r[8]=dAtof(argv[8]);
	r[9]=dAtof(argv[9]);
	r[10]=dAtof(argv[10]);
	r[11]=dAtof(argv[11]);
	r[12]=dAtof(argv[12]);
	r[13]=dAtof(argv[13]);
	dBodySetQuaternion(object->body, r);
}

ConsoleMethod(tdBody, GetRotation, const char *, 2, 2,
	"() "
	"Get a body's rotation. Returns 3x3 matrix, rows first") {
	
	argc; argv;
	char *ret = Con::getReturnBuffer(256);
	const dReal *r;

	r = dBodyGetRotation(object->body);

	dSprintf(ret, 256, "%f %f %f %f %f %f %f %f %f", r[0], r[1],
		r[2], r[4], r[5], r[6], r[8], r[9], r[10]);
	return ret;
}

ConsoleMethod(tdBody, GetQuaternion, const char *, 2, 2,
	"() "
	"Get a body's quaternion. Returns 4x3 matrix, rows first") {
	
	argc; argv;
	char *ret = Con::getReturnBuffer(256);
	const dReal *r;

	r = dBodyGetRotation(object->body);

	dSprintf(ret, 256, "%f %f %f %f %f %f %f %f %f %f %f %f",
		r[0], r[1], r[2], r[3], r[4], r[5], r[6], r[7],
		r[8], r[9], r[10], r[11]);
	return ret;
}


// Directly copied from here: http://skal.planet-d.net/demo/matrixfaq.htm

ConsoleMethod(tdBody, GetEulerAngles, const char *, 2, 2,
	"() "
	"Get a body's three euler angles") {
	
	 //  For T2D, you probably want mRadtoDeg(getWord(this, 2))

	char *ret = Con::getReturnBuffer(256);
	const dReal *mat;
	F32 angle_y, angle_x, angle_z;
	F32 trxx, tryy, C, D; // Temp variables

	mat = dBodyGetRotation(object->body);

	
	angle_y = D =  mAsin( mat[2]);		/* Calculate Y-axis angle */
	C		   =  mCos( angle_y );
	
	if ( fabs( C ) > 0.005 ) {			 /* Gimball lock? */
		trxx	  =  mat[10] / C;		   /* No, so get X-axis angle */
		tryy	  = -mat[6]  / C;
		angle_x  = mAtan( tryy, trxx );
		trxx	  =  mat[0] / C;			/* Get Z-axis angle */
		tryy	  = -mat[1] / C;
		angle_z  = mAtan( tryy, trxx );
	} else {								/* Gimball lock has occurred */
		angle_x  = 0;						 /* Set X-axis angle to zero */
		trxx	  =  mat[5];				 /* And calculate Z-axis angle */
		tryy	  =  mat[4];
		angle_z  = mAtan( tryy, trxx );
	}
	
	/* return only positive angles in [2, 2PI] */
	if (angle_x < 0) angle_x += 2 * M_PI;
	if (angle_y < 0) angle_y += 2 * M_PI;
	if (angle_z < 0) angle_z += 2 * M_PI;
	
	
	dSprintf(ret, 256, "%f %f %f", angle_x, angle_y, angle_z);
	return ret;
}


ConsoleMethod(tdBody, SetAutoDisableLinearThreshold, void, 3, 3,
	"(linear_threshold) "
	"Set threshold for Linear disabling") {

	argc; argv;
	dBodySetAutoDisableLinearThreshold(object->body, dAtof(argv[2]));
}

ConsoleMethod(tdBody, GetAutoDisableLinearThreshold, F32, 2, 2,
	"() "
	"Get threshold for Linear disabling") {

	return dBodyGetAutoDisableLinearThreshold(object->body);
}

ConsoleMethod(tdBody, SetAutoDisableAngularThreshold, void, 3, 3,
	"(Angular_threshold) "
	"Set threshold for Angular disabling") {

	argc; argv;
	dBodySetAutoDisableAngularThreshold(object->body, dAtof(argv[2]));
}

ConsoleMethod(tdBody, GetAutoDisableAngularThreshold, F32, 2, 2,
	"() "
	"Get threshold for Angular disabling") {

	return dBodyGetAutoDisableAngularThreshold(object->body);
}

ConsoleMethod(tdBody, SetAutoDisableSteps, void, 3, 3,
	"(steps) "
	"Set number of steps for disabling") {

	argc; argv;
	dBodySetAutoDisableSteps(object->body, dAtoi(argv[2]));
}

ConsoleMethod(tdBody, GetAutoDisableSteps, S32, 2, 2,
	"() "
	"Get number of steps for disabling") {

	return dBodyGetAutoDisableSteps(object->body);
}

ConsoleMethod(tdBody, SetAutoDisableTime, void, 3, 3,
	"(time) "
	"Set amount of time for disabling") {

	argc; argv;
	dBodySetAutoDisableTime(object->body, dAtof(argv[2]));
}

ConsoleMethod(tdBody, GetAutoDisableTime, F32, 2, 2,
	"() "
	"Get amount of time for disabling") {

	return dBodyGetAutoDisableTime(object->body);
}

ConsoleMethod(tdBody, SetAutoDisableFlag, void, 3, 3,
	"(do_auto_disable) "
	"Enable or disable auto disabling") {

	argc; argv;
	dBodySetAutoDisableFlag(object->body, dAtoi(argv[2]));
}

ConsoleMethod(tdBody, GetAutoDisableFlag, S32, 2, 2,
	"() "
	"Check for auto disabling") {

	return dBodyGetAutoDisableFlag(object->body);
}

ConsoleMethod(tdBody, SetAutoDisableDefaults, void, 2, 2,
	"() "
	"Set auto disabling defaults on body") {

	dBodySetAutoDisableDefaults(object->body);
}

/* TODO:

dJointID dBodyGetJoint (dBodyID, int index);

*/
