//-----------------------------------------------------------------------------
// Torque 2D. 
// Copyright (C) GarageGames.com, Inc.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Player Constants.
//-----------------------------------------------------------------------------
$stressMode = false;

$movementSpeed = 30;
$playerFireSpeed = 100;
$playerFireRate = 200;
$playerFireType = 1;
$playerMaxFireType = 4;
$playerHelpers = 0;
$playerDead = false;
$playerScore = 0;
$playerWaitUpgrade = $stressMode ? 1 : 8000;
$currentExplosions = 0;




//-----------------------------------------------------------------------------
// Player State.
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
// Setup Player.
//-----------------------------------------------------------------------------
function setupPlayer()
{
	// Create Player.
	createPlayer();
	
	// Reset Player Score.
	resetPlayerScore();
	
	// Setup Player Controls.
	setupPlayerControls();
	
	// Setup Upgrades.
	$upgradeAppearEvent = schedule( $playerWaitUpgrade + getRandom()*5000, 0, "setupUpgrades" );

	// Stress Mode?	
	if ( $stressMode )
	{
		// Yes, so add some helpers.
		addHelper();
		addHelper();
		addHelper();
		
		// Full Shields.
		shieldOn();
		shieldOn();
		
		// Max fire mode.
		$playerFireType = $playerMaxFireType;
		
		// Start Player Firing.
		playerFire();
		
		// Hide Scores so we can see debug-stats.
		lblSpaceBestScore.setVisible(false);
		lblSpaceCurrentScore.setVisible(false);
		
		// Debug Stats.
		spaceSceneGraph2D.setDebugOn( BIT(0) );
		//spaceSceneGraph2D.setDebugOn( BIT(1) );		
	}
}


//-----------------------------------------------------------------------------
// Create Player.
//-----------------------------------------------------------------------------
function createPlayer()
{
	// Generate Player.
	$player = new fxStaticSprite2D() { scenegraph = spaceSceneGraph2D; };
	$player.setGroup( 1 );
	$player.setLayer( 10 );
	$player.setPosition("-35 0");
	$player.setSize( "14 7" );
	$player.setImageMap( playershipImageMap );
	$player.setWorldLimit( clamp, "-49 -37 40 37" );
	$player.setCollisionActive( false, true );
	$player.setCollisionMaterial( standardMaterial );
	$player.setCollisionPolyCustom( 4, "-1 0 -0.1 -0.6 0.98 0.15 -0.1 0.7" );
	$player.setCollisionMasks( $enemyCollisionGroups | $upgradeCollisionGroups, $enemyCollisionLayers | $upgradeCollisionLayers );
	$player.setCollisionCallback( true );
	$player.fireLinkPoint = $player.addLinkPoint( "0.45 0.2" );
	$player.tag = "player";

	// Create Square Mount System.
	%mountSystem = new fxSceneObject2D() { scenegraph = spaceSceneGraph2D; };
	%mountSystem.setSize( 15 );
	%mountSystem.setAutoMountRotation( 180 );
	%mountSystem.mount( $player );
	$player.mountSystem = %mountSystem;
		
	// Reset Helper Slots.
	for ( %n = 0; %n < 3; %n++ )
		$playerHelperSlot[%n] = 0;

	// Attach Thruster.	
	attachThruster( $player, "-0.12 -0.33", 0);
	
	// Create Shield.
	%shield = new fxParticleEffect2D() { scenegraph = spaceSceneGraph2D; };
	%shield.loadEffect("~/client/effects/shield.eff");
	%shield.setLayer( $player.getLayer() );
	%shield.mount( $player );
	// Reference Shield.
	$player.shield = %shield;
	// Shield is off by default.
	$player.shieldOn = false;
	// Helper Shield is off by default.
	$player.helperShieldOn = false;
}


//-----------------------------------------------------------------------------
// Reset Player Score.
//-----------------------------------------------------------------------------
function resetPlayerScore()
{
	// Reset Player Score.
	$playerScore = 0;
	
	// Update Scores.
	updateScores();
}


//-----------------------------------------------------------------------------
// Add Player Score.
//-----------------------------------------------------------------------------
function addPlayerScore(%score)
{
	// Update Player Score.
	$playerScore += %score;
	
	// Update Best Score if needed.
	if ( $playerScore > $pref::spaceshooter::bestScore )
		$pref::spaceshooter::bestScore = $playerScore;
	
	// Update Scores.
	updateScores();
}



//-----------------------------------------------------------------------------
// Create and Attach Thruster.
//-----------------------------------------------------------------------------
function attachThruster(%mountObj, %mountPosition, %angle)
{
	// Create Player Thruster.	
	%thruster = new fxParticleEffect2D() { scenegraph = spaceSceneGraph2D; };
	%thruster.loadEffect("~/client/effects/smallThruster.eff");
	%thruster.setLayer( %mountObj.getLayer()+1 );
	%thruster.mount( %mountObj, %mountPosition, 0, false );
	%thruster.setRotation( %angle );
	%thruster.playEffect();
}


//-----------------------------------------------------------------------------
// Add Helper.
//-----------------------------------------------------------------------------
function addHelper()
{
	// Find a free helper slot.
	for ( %n = 0; %n < 3; %n++ )
	{
		// Free Helper Slot?
		if ( !isObject($playerHelperSlot[%n]) )
		{
			// Yes, so create helper.
			%helper = new fxStaticSprite2D() { scenegraph = spaceSceneGraph2D; };
			%helper.setGroup( 1 );
			%helper.setLayer( 10 );
			%helper.setSize( "4 2" );
			%helper.setImageMap( playerHelperImageMap );
			%helper.setCollisionActive( false, true );
			%helper.setCollisionScale("0.9 0.5");
			%helper.setCollisionMaterial( standardMaterial );
			%helper.setCollisionMasks( $enemyCollisionGroups | $upgradeCollisionGroups, $enemyCollisionLayers | $upgradeCollisionLayers );
			%helper.setCollisionCallback( true );
			%helper.fireLinkPoint = %helper.addLinkPoint( "1 0" );
			%helper.slot = %n;
			%helper.tag = "helper";

			// Attach Thruster.	
			attachThruster( %helper, "-1 0", 0 );
			
			// Calculate Helper Mount Position.
			%mountPosition = (mCos(mDegToRad(120*%n))) SPC (mSin(mDegToRad(120*%n)));
			// Mount.
			%helper.mount( $player.mountSystem, %mountPosition );
			
			// Create Helper Shield.
			%shield = new fxParticleEffect2D() { scenegraph = spaceSceneGraph2D; };
			%shield.loadEffect("~/client/effects/shield.eff");
			%shield.setLayer( %helper.getLayer() );
			%shield.mount( %helper );
			// Shield Off by default.
			%helper.shieldOn = false;
			// Reference Shield.
			%helper.shield = %shield;
			
			// Are the helpers shields on?
			if ( $player.helperShieldOn )
			{
				// Flag Shield On.
				%helper.shieldOn = true;
				// Player Shield Effect.
				%shield.playEffect();
			}
			
			// Reference Helper Slot.
			$playerHelperSlot[%n] = %helper;
			
			// Increase Helper Count.
			$playerHelpers++;
			
			// Finish Here.
			return;
		}
	}
}


//-----------------------------------------------------------------------------
// Remove Helper.
//-----------------------------------------------------------------------------
function removeHelper(%slot)
{
	// Is there a helper object here?
	if ( isObject($playerHelperSlot[%slot]) )
	{
		// Delete Helper.
		$playerHelperSlot[%slot].safeDelete();
		// Free Slot.
		$playerHelperSlot[%slot] = -1;
		
		// Decrease Helper Count.
		$playerHelpers--;	
	}
}


//-----------------------------------------------------------------------------
// Create Explosion Effect.
//-----------------------------------------------------------------------------
function createExplosion( %object )
{
	// Cap number of explosions.
	if ( $currentExplosions >= $maxExplosions )
		return;
		
	// Ignore if object not around anymore.
	if ( !isObject(%object) )
		return;	
		
	// Shockwave Explosion.	
	%explosion = new fxParticleEffect2D() { scenegraph = spaceSceneGraph2D; };
	%explosion.loadEffect("t2d/client/effects/shockwave_burst.eff");
	%explosion.setPosition( %object.getPosition() );
	%explosion.setLayer( %object.getLayer()+1 );
	%explosion.setEffectLifeMode( kill, 0.5 );
	%explosion.playEffect();
	%explosion.tag = "explosion";
	
	// Explosion Audio.
	alxPlay( explosionAudio );
   $currentExplosions++;
}


//-----------------------------------------------------------------------------
// Create Boss Explosion Effect.
//-----------------------------------------------------------------------------
function createBossExplosion( %object, %offset )
{
	// Cap number of explosions.
	if ( $currentExplosions >= $maxExplosions )
		return;
	
	// Ignore if object not around anymore.
	if ( !isObject(%object) )
		return;
		
	// Shockwave Explosion.	
	%explosion = new fxParticleEffect2D() { scenegraph = spaceSceneGraph2D; };
	%explosion.loadEffect("t2d/client/effects/shockwave_burst.eff");
	%explosion.setPosition( vectorAdd2D( %object.getPosition(), %offset ) );
	%explosion.setLayer( %object.getLayer()+1 );
	%explosion.setEffectLifeMode( kill, 0.5 );
	%explosion.playEffect();
	
	// Explosion Audio.
	alxPlay( explosionAudio );
   $currentExplosions++;
}


//-----------------------------------------------------------------------------
// Player Shield On.
//-----------------------------------------------------------------------------
function shieldOn()
{
	// Ignore if both shields are already on.
	if ( $player.shieldOn && $player.helperShieldOn )
		return;

	// Are the players shields already on?
	if ( $player.shieldOn )
	{
		// Yes, so turn the helper shields on.
		$player.helperShieldOn = true;
		
		// Active Shields on Helpers.		
		for ( %n = 0; %n < 3; %n++ )
		{
			// Free Helper Slot?
			if ( isObject($playerHelperSlot[%n]) )
			{
				// Flag Shield On.
				$playerHelperSlot[%n].shieldOn = true;
				// Play Shield Effect.
				$playerHelperSlot[%n].shield.playEffect();
			}
		}
	}
	else
	{
		// Flag Shield On.
		$player.shieldOn = true;
	
		// Play Shield Effect.
		$player.shield.playEffect();
	}
	
	// Shield On Audio.
	alxPlay( upgrade3PickupAudio );
}


//-----------------------------------------------------------------------------
// Player Shield Off.
//-----------------------------------------------------------------------------
function shieldOff()
{
	// Ignore if shield is already off.
	if ( !$player.shieldOn )
		return;
		
	// Flag Player Shield Off.
	$player.shieldOn = false;

	// Stop Shield Effect.
	$player.shield.stopEffect(true, false);
	
	// Turn off the helper shields as well.
	helperShieldOff();
	
	// Shield Off Audio.
	alxPlay( upgrade3LooseAudio );
}


//-----------------------------------------------------------------------------
// Helper Shield Off.
//-----------------------------------------------------------------------------
function helperShieldOff()
{
	// Ignore if the helper shields are already off.
	if ( !$player.helperShieldOn )
		return;
		
	// Flag Helper Shield Off.
	$player.helperShieldOn = false;
	
	// Deactivate Shields on Helpers.		
	for ( %n = 0; %n < 3; %n++ )
	{
		// Free Helper Slot?
		if ( isObject($playerHelperSlot[%n]) )
		{
			// Yes, so flag Shield Off.
			$playerHelperSlot[%n].shieldOn = false;
			// Stop Shield Effect.
			$playerHelperSlot[%n].shield.stopEffect(true, false);
		}
	}
}


//-----------------------------------------------------------------------------
// Setup Player Controls.
//-----------------------------------------------------------------------------
function setupPlayerControls()
{
		// Regenerate Player Map.
	if( isObject(playerMap) )
	   playerMap.delete();
	
	// Create a new map.
	new ActionMap(playerMap);
	
	// Create Player Map.
	playerMap.bindCmd(keyboard, "w", "playerUp();", "playerUpStop();");
	playerMap.bindCmd(keyboard, "up", "playerUp();", "playerUpStop();");
	playerMap.bindCmd(keyboard, "s", "playerDown();", "playerDownStop();");
	playerMap.bindCmd(keyboard, "down", "playerDown();", "playerDownStop();");
	playerMap.bindCmd(keyboard, "a", "playerLeft();", "playerLeftStop();");
	playerMap.bindCmd(keyboard, "left", "playerLeft();", "playerLeftStop();");	
	playerMap.bindCmd(keyboard, "i", "playerLeft();", "playerLeftStop();");
	playerMap.bindCmd(keyboard, "d", "playerRight();", "playerRightStop();");
	playerMap.bindCmd(keyboard, "right", "playerRight();", "playerRightStop();");	
	playerMap.bindCmd(keyboard, "o", "playerRight();", "playerRightStop();");   
	playerMap.bindCmd(keyboard, "p", "scenePauseToggle();", "");

	// Stress Mode?	
	if ( !$stressMode )
		// No, so allow firing.
		playerMap.bindCmd(keyboard, "space", "playerFire();", "playerFireStop();");
	
	// Activate Player Controls.
	playerMap.push();
}


//-----------------------------------------------------------------------------
// Setup Upgrades.
//-----------------------------------------------------------------------------
function setupUpgrades()
{
	if ( !$playerDead && !$spaceScenePauseFlag )
	{
		// Choose Upgrade Type.
		%type = mFloor(getRandom()*3)+1;
		
		// Create Ship Type.
		%upgrade = new fxStaticSprite2D() { scenegraph = spaceSceneGraph2D; };
		%upgrade.setImageMap( "playerUpgrade" @ %type @ "ImageMap" );
		%upgrade.setPosition( "53" SPC (-20 + getRandom()*40) );
		%upgrade.setSize( "4 4" );
		%upgrade.setGroup( 5 );
		%upgrade.setLayer( 5 );
		%upgrade.setWorldLimit( kill, "-53 -40 60 50" );
		%upgrade.setCollisionActive( true, true );
		%upgrade.setCollisionMaterial( standardMaterial );
		%upgrade.setCollisionPolyPrimitive( 10 );
		%upgrade.setCollisionMasks( $playerCollisionGroups, $playerCollisionLayers );
		%upgrade.setCollisionCallback( true );
		%upgrade.setLinearVelocityX( -40 );
		%upgrade.tag = "upgrade";
		%upgrade.upgradeType = %type;
		
		// Play Audio.
		%upgrade.audio = alxPlay( upgradeAppearAudio );
	}
		
	// Reschedule an upgrade.
	$upgradeAppearEvent = schedule( $playerWaitUpgrade + getRandom()*5000, 0, "setupUpgrades" );
}

//-----------------------------------------------------------------------------
// Player Up.
//-----------------------------------------------------------------------------
function playerUp()
{
	// Set it moving up.
	$player.setLinearVelocityY( -$movementSpeed );
}


//-----------------------------------------------------------------------------
// Player Down.
//-----------------------------------------------------------------------------
function playerDown()
{
	// Set it moving down.
	$player.setLinearVelocityY( $movementSpeed );
}


//-----------------------------------------------------------------------------
// Player Left.
//-----------------------------------------------------------------------------
function playerLeft()
{
	// Set it moving left.
	$player.setLinearVelocityX( -$movementSpeed );
}


//-----------------------------------------------------------------------------
// Player Right.
//-----------------------------------------------------------------------------
function playerRight()
{
	// Set it moving right.
	$player.setLinearVelocityX( $movementSpeed );
}


//-----------------------------------------------------------------------------
// Player Up (Stop).
//-----------------------------------------------------------------------------
function playerUpStop()
{
	// If we're moving up then nullify any upward movement.
	if ( $player.getLinearVelocityY() < 0 )
		$player.setLinearVelocityY( 0 );	
}


//-----------------------------------------------------------------------------
// Player Down (Stop).
//-----------------------------------------------------------------------------
function playerDownStop()
{
	// If we're moving down then nullify any downward movement.
	if ( $player.getLinearVelocityY() > 0 )
		$player.setLinearVelocityY( 0 );	
}


//-----------------------------------------------------------------------------
// Player Left (Stop).
//-----------------------------------------------------------------------------
function playerLeftStop()
{
	// If we're moving left then nullify any leftward movement.
	if ( $player.getLinearVelocityX() < 0 )
		$player.setLinearVelocityX( 0 );	
}


//-----------------------------------------------------------------------------
// Player Right (Stop).
//-----------------------------------------------------------------------------
function playerRightStop()
{
	// If we're moving right then nullify any rightward movement.
	if ( $player.getLinearVelocityX() > 0 )
		$player.setLinearVelocityX( 0 );	
}


//-----------------------------------------------------------------------------
// Player Fire.
//-----------------------------------------------------------------------------
function playerFire()
{
	// Is the Player Dead?
	if ( !$playerDead && !$spaceScenePauseFlag )
	{	
		// No, so handle Player Projectile Type.
		switch( $playerFireType )
		{
			// Default Single Projectile.
			case 1:
				firePlayerProjectile(0, true);
	
			// Projectile (Upgrade Stage #1).
			case 2:			
				%proj1 = firePlayerProjectile(-10, true);
				%proj2 = firePlayerProjectile(0, true);
				%proj3 = firePlayerProjectile(10, true);
				
			// Projectile (Upgrade Stage #2).
			case 3:			
				%proj1 = firePlayerProjectile(-20, false);
				%proj2 = firePlayerProjectile(-10, true);
				%proj3 = firePlayerProjectile(0, true);
				%proj4 = firePlayerProjectile(10, true);
				%proj5 = firePlayerProjectile(20, false);
				
			// Projectile (Upgrade Stage #3).
			case 4:			
				%proj1 = firePlayerProjectile(-20, false);
				%proj2 = firePlayerProjectile(-10, true);
				%proj3 = firePlayerProjectile(0, true);
				%proj4 = firePlayerProjectile(10, true);
				%proj5 = firePlayerProjectile(20, true);
				
				%proj6 = firePlayerProjectile(80, false);				
				%proj7 = firePlayerProjectile(100, false);
				
				%proj8 = firePlayerProjectile(-80, false);				
				%proj9 = firePlayerProjectile(-100, false);
		}
		
		// Player Fire Audio.
		alxPlay( playerFireAudio );
	}
		
	// Reschedule another fire.
	$playerFireSchedule = schedule( $playerFireRate, 0, "playerFire" );
}



//-----------------------------------------------------------------------------
// Player Fire (Stop)
//-----------------------------------------------------------------------------
function playerFireStop()
{
	// Stop Firing event if active.
	if ( isEventPending($playerFireSchedule) )
		cancel( $playerFireSchedule );
}


//-----------------------------------------------------------------------------
// Fire Player Projectile.
//-----------------------------------------------------------------------------
function firePlayerProjectile( %angle, %addToHelper )
{
	// Calculate Velocity.
	%velocity = ($playerFireSpeed*mCos(mDegToRad(%angle))) SPC ($playerFireSpeed*mSin(mDegToRad(%angle)));
	
	// Player Fire.
	createPlayerProjectile( $player, %velocity, %angle );

	// Finish here if we're not adding to the helpers.
	if ( !%addToHelper )
		return;
		
	// Fire from Helpers.
	for ( %n = 0; %n < 3; %n++ )
		// Helper?
		if ( isObject($playerHelperSlot[%n]) )
			// Yes, so fire from it.
			createPlayerProjectile( $playerHelperSlot[%n], %velocity, %angle );
	
}


//-----------------------------------------------------------------------------
// Create Player Projectile.
//-----------------------------------------------------------------------------
function createPlayerProjectile( %object, %velocity, %angle )
{
	// Create Projectile.
	%projectile = new fxStaticSprite2D() { scenegraph = spaceSceneGraph2D; };
	%projectile.setGroup( 2 );
	%projectile.setLayer( 9 );
	%projectile.setPosition( %object.getLinkPoint(%object.fireLinkPoint) );
	%projectile.setSize( "3 1.5" );
	%projectile.setImageMap( playermissileImageMap );
	%projectile.setRotation( %angle );
	%projectile.setWorldLimit( kill, "-52 -40 52 40" );
	%projectile.tag = "playerfire";
	// Collisions.
	%projectile.setCollisionActive(true, true);
	%projectile.setCollisionMasks( $enemyCollisionGroups, $enemyCollisionLayers );
	%projectile.setCollisionScale("0.9 0.5");
	%projectile.setCollisionMaterial(projectileMaterial);
	%projectile.setCollisionCallback( true );
	// Movement.
	%projectile.setLinearVelocity( %velocity );
	
	// Return Projectile.
	return %projectile;
}


//-----------------------------------------------------------------------------
// Create Player Debris.
//-----------------------------------------------------------------------------
function createPlayerDebris()
{
	// Create Player Thruster.	
	%debris = new fxParticleEffect2D() { scenegraph = spaceSceneGraph2D; };
	%debris.loadEffect( "~/client/effects/playerDebris.eff");
	%debris.setPosition( $player.getPosition() );
	%debris.setLayer( $player.getLayer() );
	%debris.setEffectLifeMode( kill, 3 );
	%debris.playEffect();	
}


//-----------------------------------------------------------------------------
// Game Collisions.
//-----------------------------------------------------------------------------
function fxSceneObject2D::onCollision( %srcObj, %dstObj, %srcRef, %dstRef, %time, %normal, %contactCount, %contacts )
{
	// Handle Source Object.
	switch$( %srcObj.tag )
	{
		// Player Hit something.
		case "player":
		
			// Upgrade?
			if ( %dstObj.tag $= "upgrade" )
			{
				// Fire Upgrade?
				if ( %dstObj.upgradeType == 1 )
				{
					// Yes, so upgrade fire!
					if ( $playerFireType < $playerMaxFireType )
						$playerFireType++;
						
					// Player Upgrade#1 Audio.
					alxPlay( upgrade1PickupAudio );
					
				}
				// Helper Upgrade?
				else if ( %dstObj.upgradeType == 2 )
				{
					// Yes, so add helper.
					addHelper();

					// Player Upgrade#1 Audio.
					alxPlay( upgrade2PickupAudio );

					// Are we loaded for bear? (ignore shields).
					// NOTE:- We'll limit the number of bosses so it doesn't get too silly!
					if ( $playerFireType == $playerMaxFireType && $playerHelpers > 1 && $bossCount < 2 )
					{
						// Yes, so give us a boss!
						constructEnemyShip( 3, "80 0", "-30 0", 20, 30, 0 );
						
						// Play Ready #2 Audio.
						alxPlay( ready2Audio );
					}
						
					
				}
				// Shield Upgrade?
				else if ( %dstObj.upgradeType == 3 )
				{
					// Yes, so turn-on shield.
					shieldOn();
				}
				
				// Destroy Upgrade Object.
				%dstObj.safeDelete();
			}
			// Enemy or Enemy Fire?
			else if ( %dstObj.tag $= "enemyship" || %dstObj.tag $= "enemyfire" )
			{
				// Stress Mode?	
				if ( $stressMode )
					// Yes, so invincible.
					return;

				// Is the shield on?
				if ( $player.shieldOn )
				{
					// Yes, so turn shield off.
					shieldOff();
				}
				else
				{
					// No, so kill player.
					killPlayer();
					
					// Schedule new Player.
					schedule( 3000, 0, "newPlayer");
				}
			}
			
			
		// Player Fire.
		case "playerfire":
			if ( %dstObj.tag !$= "enemyfire" &&  %dstObj.tag !$= "upgrade" )
				// Delete the Player Projectile.
				%srcObj.safeDelete();
		

		// Player Helper.
		case "helper":
			// Stress Mode?	
			if ( $stressMode )
				// Yes, so invincible.
				return;
		
			// Helper Shield On?
			if ( %srcObj.shieldOn )
			{
				// Yes, so flag Shield Off.
				%srcObj.shieldOn = false;
				// Stop Shield Effect.
				%srcObj.shield.stopEffect(true, false);
				// Shake Camera.
				spaceScrollerSceneWindow2D.startCameraShake( 5, 0.5 );
			}
			else
			{
				// No, so create Explosion.
				createExplosion( %srcObj );
				// Remove Helper.
				removeHelper(%srcObj.slot);
				// Shake Camera.
				spaceScrollerSceneWindow2D.startCameraShake( 10, 0.5 );
			}
			
		
		// Enemy Hit something.
		case "enemyship":
			// Create Explosion.
			createExplosion( %srcObj );
			// Sometimes Create Debris.
			if ( getRandom() < 0.3 )
				createEnemyDebris( %srcObj );
			// Destroy Enemy Ship.			
			%srcObj.waveform.safeDelete();
			
			
			// Add Player Score.
			addPlayerScore( (8 - $playerHelpers*2) + (10 - ($playerFireType*2)) ); 	
			
		// Enemy Boss something.
		case "enemyboss":
			// Sometimes Create Explosion/Debris.
			if ( getRandom() < 0.1 )
			{
				createExplosion( %dstObj );
				createEnemyDebris( %srcObj );
			}

			// Decrease Boss Shields.
			%srcObj.shields--;
				
			// Destroyed?
			if ( %srcObj.shields < 0 )
			{
				// Shake Camera.
				spaceScrollerSceneWindow2D.startCameraShake( 20, 3 );			
				
				// Create Boss Explosion.
				for ( %n = 0; %n < 16; %n++ )
					createBossExplosion( %srcObj, (-30+getRandom()*60) SPC (-15+getRandom()*30) );
					
				// Destroy Enemy Ship.			
				%srcObj.waveform.safeDelete();
				
				// Add Player Score.
				addPlayerScore( 5000 ); 	
			}
			

			
		// Enemy Fire.
		case "enemyfire":
			// Delete (unless hit by player fire).
			if ( %dstObj.tag !$= "playerFire" )
				// Delete the Enemy Projectile.
				%srcObj.safeDelete();
			
	}
}


//-----------------------------------------------------------------------------
// Kill Player.
//-----------------------------------------------------------------------------
function killPlayer()
{
	// Deactivate Controls.
	playerMap.pop();

	// Player Shield Off.
	shieldOff();
	
	// Remove All the Helpers.
	for ( %n = 0; %n < 3; %n++ )
	{
		// Create Explosion at Helper.
		createExplosion( $playerHelperSlot[%n] );
		// Remove Helper.
		removeHelper( %n );
	}

	// Shake Camera.
	spaceScrollerSceneWindow2D.startCameraShake( 30, 2 );

	// Create Explosion at Helper.
	createExplosion( $player );
	// Create Player Debris.
	createPlayerDebris();

	
	// Reset Player Start Position.					
	$player.setPosition("-35 0");
	// Reset Velocity.
	$player.setAtRest();
	// Disable Player.
	$player.setEnabled( false );

	// Reset Fire Type.
	$playerFireType = 1;
	// Flag Player Dead.
	$playerDead = true;

	// Fast-Forward Enemy Stuff.
	fastForwardEnemyStuff();
	
	// Stop Upgrade Appear Event.
	if ( isEventPending($upgradeAppearEvent) )
		cancel( $upgradeAppearEvent );
			
	// Stop Boss Appear Event.
	if ( isEventPending($bossAppearEvent) )
		cancel( $bossAppearEvent );	
}


//-----------------------------------------------------------------------------
// New Player.
//-----------------------------------------------------------------------------
function newPlayer()
{
	// Enable Player.
	$player.setEnabled( true );
	
	// Flag Player Alive.
	$playerDead = false;
	
	// Reset Player Score.
	resetPlayerScore();
	
	// Activate Player Controls.
	playerMap.push();
	
	// Player Start.
	alxPlay( ready1Audio );

	// Reschedule an upgrade.
	$upgradeAppearEvent = schedule( $playerWaitUpgrade + getRandom()*5000, 0, "setupUpgrades" );
	
	// Schedule a boss.
	$bossAppearEvent = schedule( $bossAppearRate, 0, "spawnBoss" );
}


//-----------------------------------------------------------------------------
// Objects Deleted.
//-----------------------------------------------------------------------------
function fxSceneGraph2D::onSafeDelete( %scene, %object )
{
	// Handle the death of game objects.
	switch$( %object.tag )
	{
		// Explostions.
		case "explosion":
			// Decrease Explosion Count.
			$currentExplosions--;
		
			
		// Upgrades.
		case "upgrade":
			// Stop Upgrade Sound.
			alxStop( %object.audio );

			
		// Debris.
		case "enemydebris":
			// Decrease Enemy Debris Count.
			$debrisCount--;
			
		
		// Enemy Ship.
		case "enemyship":
			// Decrease Enemy Count.
			$enemyCount--;
			// Clamp Enemy Count.
			if ( $enemyCount < 0 )
				$enemyCount = 0;	
		
		// Enemy Boss.
		case "enemyboss":
			// Decrease Boss Count.
			$bossCount--;
			// Clamp Boss Count.
			if ($bossCount < 0 )
				$bossCount = 0;

	}
}



