//-----------------------------------------------------------------------------
// Torque 2D. 
// Copyright (C) GarageGames.com, Inc.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Enemy Constants.
//-----------------------------------------------------------------------------
$enemyStartPointX = 70;
$enemyMaxFireSpeed = 25;
$enemyFireSpeed = 100;
$bossAppearRate = $stressMode ? 1000 : (1000 * 45);
$bossFireRate = 800;
$bossFireSpeed = 40;
$bossCount = 0;
$enemyCount = 0;
$debrisCount = 0;


// Limits for Enemy/Bosses/Debris/Explosions.
$maxBossCount = 2;
$maxEnemyCount = 15;
$maxDebris = 5;
$maxExplosions = 5;

//-----------------------------------------------------------------------------
// Setup Enemy.
//-----------------------------------------------------------------------------
function setupEnemy()
{
	// Create Active Enemy Set.
	$activeEnemyStuff = new SimSet();
	
	// Schedule Enemy.
	schedule( 1000, 0, "spawnEnemyWave" );
	
	// Schedule a boss.
	$bossAppearEvent = schedule( $bossAppearRate, 0, "spawnBoss" );
}


//-----------------------------------------------------------------------------
// Spawn Enemy Wave.
//-----------------------------------------------------------------------------
function spawnEnemyWave()
{
	// Is the Player Dead?
	if ( !$playerDead && !$spaceScenePauseFlag )
	{	
		// No, so choose Enemy Ship Type.
		%shipType = mFloor(getRandom()*2)+1;
		
		// Calculate Start Point.
		%startPoint = $enemyStartPointX SPC (-30 + (getRandom() * 60));
		
		// Calculate Max Velocity based upon players fire-power.
		%maxVelocity = 30 + $playerFireType*20;
		
		// Calculate Velocity.
		%velocity = (-20 - getRandom()*%maxVelocity) SPC 0;
		
		// Calculate Waveform Amplitude/Speed.
		%waveformAmp = 5 + getRandom() * 20;
		%waveformSpeed = 30 + getRandom()*100;
		
		// Create Lead Ship.
		%leadShip = constructEnemyShip( %shipType, %startPoint, %velocity, %waveformAmp, %waveformSpeed, 0 );
		
		// Calculate Max Enemy Count based upon players helper count.
		%maxEnemy = 1+($playerHelpers*2);
		// Choose Enemy Count.
		%enemyCount = mFloor(getRandom()*4)+%maxEnemy;

		// Set Parent Ship.
		%parentShip = %leadShip;
		
		// Create Enemy Formation (all set to follow the lead ship).
		for ( %n = 0; %n < %enemyCount; %n++ )
		{
			// Create Formation Ship.
			%enemyShip = constructEnemyShip( %shipType, %startPoint, %velocity, %waveformAmp, %waveformSpeed, %n+1 );
			// Update Parent Ship.
			%parentShip = %enemyShip;
		}
	}
	
	// Calculate Next-Spawn based upon players fire-power/helpers.
	%nextSpawn = 3000 - ($playerFireType*333) - ($playerHelpers*333);
	
	// Schedule Enemy.
	schedule( %nextSpawn, 0, "spawnEnemyWave" );
}


//-----------------------------------------------------------------------------
// Construct Boss.
//-----------------------------------------------------------------------------
function spawnBoss()
{
   if ($bossCount >= $maxBossCount)
      return;
   // Give us a boss!
	constructEnemyShip( 3, "80 0", "-30 0", 20, 30, 0 );
	
	// Play Ready #2 Audio.
	alxPlay( ready2Audio );
	
	// Schedule a boss.
	$bossAppearEvent = schedule( $bossAppearRate, 0, "spawnBoss" );
}


//-----------------------------------------------------------------------------
// Construct Ship Helper.
//-----------------------------------------------------------------------------
function constructEnemyShip( %type, %position, %velocity, %waveformAmp, %waveformSpeed, %waveformIndex )
{
   if (($bossCount + $enemyCount) > $maxEnemyCount)  // OBOB city, but who cares?  Just want a rough limit to the total number of enemies.
      return 0;
   // Construct Appropriate Ship Type.
	switch( %type )
	{
		// Type #1.
		case 1:
			return constructEnemyShipType1( %position, %velocity, %waveformAmp, %waveformSpeed, %waveformIndex );
			
		// Type #2.
		case 2:
			return constructEnemyShipType2( %position, %velocity, %waveformAmp, %waveformSpeed, %waveformIndex );
		
		// Boss.
		case 3:
			return constructEnemyBoss( %position, %velocity, %waveformAmp, %waveformSpeed, %waveformIndex );
			
		// Unknown Type!
		default:
			// Invalid Enemy Ship Type.
			echo( "SpaceScroller - Invalid Enemy Ship Type" SPC %type SPC "@" SPC %position );
	}
}


//-----------------------------------------------------------------------------
// Construct Enemy Ship Type #1.
//-----------------------------------------------------------------------------
function constructEnemyShipType1( %position, %velocity, %waveformAmp, %waveformSpeed, %waveformIndex )
{
	// Create Ship Type.
	%enemy = new fxStaticSprite2D() { scenegraph = spaceSceneGraph2D; };
	%enemy.setImageMap( enemyship1ImageMap );
	%enemy.setSize( "14 7" );
	%enemy.setGroup( 3 );
	%enemy.setLayer( 15 );
	%enemy.setCollisionActive( true, true );
	%enemy.setCollisionMaterial( standardMaterial );
	%enemy.setCollisionPolyCustom( 5, "-0.9 0 0 -0.6 1 -0.3 1 0.3 0 0.5" );
	%enemy.setCollisionMasks( $playerCollisionGroups, $playerCollisionLayers );
	%enemy.setCollisionCallback( true );
	%enemy.shipType = 1;
	%enemy.tag = "enemyship";	
	
	// Choose whether to fire torpedo and at what time. 	
	if ( getRandom() < 0.75 )
		%enemy.schedule( 250 + mFloor(getRandom()*1500), "enemyFireTorpedo" );

	// Schedule a potential missile fire. 	
	%enemy.schedule( 250 + mFloor(getRandom()*1500), "enemyFireMissle" );
		
	// Attach Thruster.	
	attachThruster( %enemy, "0.8 -0.12", 180 );
	attachThruster( %enemy, "0.8 0.18", 180 );
	
	// Create and attach to waveform.
	%waveform = createWaveformObject( %enemy, %position, %velocity, %waveformAmp, %waveformSpeed, %waveformIndex );
	
	// Increase Enemy Count.
	$enemyCount++;
	
	// Return Enemy.
	return %enemy;
}


//-----------------------------------------------------------------------------
// Construct Enemy Ship Type #2.
//-----------------------------------------------------------------------------
function constructEnemyShipType2( %position, %velocity, %waveformAmp, %waveformSpeed, %waveformIndex )
{
   // Create Ship Type.
	%enemy = new fxStaticSprite2D() { scenegraph = spaceSceneGraph2D; };
	%enemy.setImageMap( enemyship2aImageMap );
	%enemy.setSize( "14 7" );
	%enemy.setGroup( 3 );
	%enemy.setLayer( 15 );
	%enemy.setCollisionActive( true, true );
	%enemy.setCollisionMaterial( standardMaterial );
	%enemy.setCollisionPolyCustom( 5, "-0.7 0 -0.34 -0.75 0.2 -0.9 0.65 -0.5 0.17 0.9" );
	%enemy.setCollisionMasks( $playerCollisionGroups, $playerCollisionLayers );
	%enemy.setCollisionCallback( true );
	%enemy.shipType = 2;
	%enemy.tag = "enemyship";
	
	// Choose whether to fire torpedo and at what time. 	
	if ( getRandom() < 0.75 )
		%enemy.schedule( 1000 + mFloor(getRandom()*3000), "enemyFireTorpedo" );
	
	// Schedule a potential missile fire. 	
	%enemy.schedule( 1000 + mFloor(getRandom()*3000), "enemyFireMissle" );
	
	// Create Attached Engine.
	%engine = new fxStaticSprite2D() { scenegraph = spaceSceneGraph2D; };
	%engine.setImageMap( enemyship2bImageMap );
	%engine.setSize( "8 4" );
	%engine.setLayer( 15 );
	%engine.mount( %enemy, "0.04 -0.25", 0, false );

	// Attach Thruster.	
	attachThruster( %engine, "0.4 0", 180 );
	
	// Create and attach to waveform.
	%waveform = createWaveformObject( %enemy, %position, %velocity, %waveformAmp, %waveformSpeed, %waveformIndex );
	
	// Increase Enemy Count.
	$enemyCount++;
	
	// Return waveform itself.
	return %waveform;
}


//-----------------------------------------------------------------------------
// Construct Boss Ship
//-----------------------------------------------------------------------------
function constructEnemyBoss( %position, %velocity, %waveformAmp, %waveformSpeed, %waveformIndex )
{
	// Create Ship Type.
	%enemy = new fxStaticSprite2D() { scenegraph = spaceSceneGraph2D; };
	%enemy.setImageMap( enemyship3aImageMap );
	%enemy.setSize( "50 25" );
	%enemy.setGroup( 3 );
	%enemy.setLayer( 15 );
	%enemy.setCollisionActive( true, true );
	%enemy.setCollisionMaterial( standardMaterial );
	%enemy.setCollisionPolyCustom( 5, "-0.7 0 -0.34 -0.75 0.2 -0.9 0.65 -0.5 0.17 0.9" );
	%enemy.setCollisionMasks( $playerCollisionGroups, $playerCollisionLayers );
	%enemy.setCollisionCallback( true );
	%enemy.shipType = 3;
	%enemy.tag = "enemyboss";
	%enemy.shields = 350;
	
	// Create Attached Engine.
	%engine = new fxStaticSprite2D() { scenegraph = spaceSceneGraph2D; };
	%engine.setImageMap( enemyship3bImageMap );
	%engine.setSize( "20 10" );
	%engine.setLayer( 15 );
	%engine.mount( %enemy, "0.5 0.1", 0, false );

	// Attach Thruster.	
	attachThruster( %engine, "0.35 -0.2", 160 );
	attachThruster( %engine, "0.35 0", 180 );
	attachThruster( %engine, "0.35 0.2", 200 );

	// Create Attached Gun.
	%gun = new fxStaticSprite2D() { scenegraph = spaceSceneGraph2D; };
	%gun.setImageMap( enemyship3cImageMap );
	%gun.setSize( "12 10" );
	%gun.setLayer( 15 );
	%gun.mount( %enemy, "-0.38 0.3", 0, false );
	%gun.setAutoRotation( 180 );

	// Calculate Boss Guns.	
	%enemy.bossGun1 = %gun.addLinkPoint("-0.3 0");
	%enemy.bossGun2 = %gun.addLinkPoint("0.3 0");
	// Reference Gun.
	%enemy.bossGun = %gun;
	
	// Schedule boss gun-fire. 	
	%enemy.schedule( $bossFireRate, "bossFireTorpedo" );
	
	// Create and attach to waveform.
	%waveform = createWaveformObject( %enemy, %position, %velocity, %waveformAmp, %waveformSpeed, %waveformIndex );
	
	// Increase Boss Count.
	$bossCount++;
	
	// Return waveform itself.
	return %waveform;
}


//-----------------------------------------------------------------------------
// Create Waveform Object.
//-----------------------------------------------------------------------------
function createWaveformObject( %enemyShip, %position, %velocity, %waveformAmp, %waveformSpeed, %waveformIndex )
{
	// Adjust Position by Waveform Index.
	%position = vectorAdd2D( %position, (%waveformIndex*10) SPC "0" );
	
	// Create Waveform.
	%waveform = new fxSceneObject2D() { scenegraph = spaceSceneGraph2D; };
	%waveform.setPosition( %position );
	%waveform.setSize( %waveformAmp );
	%waveform.setLinearVelocityX( %velocity );
	%waveform.tag = "waveform";
	
	// Is this a standard ship?
	if ( %enemyShip.shipType < 3 )
		// Yes, so set world limit outside of window.
		%waveform.setWorldLimit( kill, "-100 -100 1000 100" );
	else
		// No, it's a boss so limit to center of window.
		%waveform.setWorldLimit( clamp, "10 -37.5 1000 37.5" );

	%waveform.setRotation( %waveformIndex * 40 );
	%waveform.setAutoRotation( %waveformSpeed );
	
	// Mount Enemy-Ship onto Waveform.
	%enemyShip.mount( %waveform, "1 0", 2, false );
	
	// Setup References.
	%enemyShip.waveform = %waveform;
	%waveform.enemyShip = %enemyShip;

	// Add to Active Enemy Stuff.
	$activeEnemyStuff.add( %waveform );
	
	// Return Waveform object.
	return %waveform;
}


//-----------------------------------------------------------------------------
// Create Enemy Debris.
//-----------------------------------------------------------------------------
function createEnemyDebris( %object )
{
	if ($debrisCount > $maxDebris)
         return 0;
   // Create Player Thruster.	
	%debris = new fxParticleEffect2D() { scenegraph = spaceSceneGraph2D; };
	%debris.loadEffect( "~/client/effects/enemyDebris" @ mFloor(1+getRandom()*2) @ ".eff");
	%debris.setPosition( %object.getPosition() );
	%debris.setLayer( %object.getLayer() );
	%debris.setEffectLifeMode( kill, 1 );
	%debris.playEffect();	
	%debris.tag = "enemydebris";
   $debrisCount++;
}


//-----------------------------------------------------------------------------
// Enemy Fire (Torpedo)
//-----------------------------------------------------------------------------
function fxStaticSprite2D::enemyFireTorpedo( %object )
{
   // Is the Player Dead?
	if ( !$playerDead && !$spaceScenePauseFlag )
	{	
		// No, so calculate Speed Vector to Player.
		%fireVec = vectorScale2D( vectorNormalise2D( vectorSub2D( $player.getPosition(), %object.getPosition() ) ), $enemyMaxFireSpeed );
		
		// Don't fire backwards!
		if ( %fireVec > 0 )
			return;
			
		// Create Projectile.
		%enemyFire = new fxSceneObject2D() { scenegraph = spaceSceneGraph2D; };
		%enemyFire.setPosition( %object.getPosition() );
		%enemyFire.setSize( "1 1" );
		%enemyFire.setGroup( 4 );
		%enemyFire.setLayer( 14 );
		%enemyFire.setWorldLimit( kill, "-52 -40 52 40" );
		%enemyFire.setCollisionActive( true, true );
		%enemyFire.setCollisionMaterial( standardMaterial );
		%enemyFire.setCollisionPolyPrimitive( 6 );
		%enemyFire.setCollisionMasks( $playerCollisionGroups, $playerCollisionLayers );
		%enemyFire.setCollisionCallback( true );
		%enemyFire.setLinearVelocity( %fireVec );
		%enemyFire.tag = "enemyfire";
	
		// Fire FX.
		%fireFX = new fxParticleEffect2D() { scenegraph = spaceSceneGraph2D; };
		%fireFX.loadEffect("~/client/effects/torpedo.eff");
		%fireFX.setLayer( %object.getLayer() );
		%fireFX.setGroup( 14 );
		%fireFX.mount( %enemyFire );
		%fireFX.playEffect();
		
		// Add to Active Enemy Stuff.
		$activeEnemyStuff.add( %enemyFire );
		
		// Torpedo Fire Audio.
		alxPlay( enemyTorpedoFireAudio );		
	}
}


//-----------------------------------------------------------------------------
// Enemy Fire (Missile)
//-----------------------------------------------------------------------------
function fxStaticSprite2D::enemyFireMissle( %object )
{
	// Is the Player Dead?
	if ( !$playerDead && !$spaceScenePauseFlag && getRandom() < 0.75 )
	{	
		// Create Projectile.
		%enemyFire = new fxStaticSprite2D() { scenegraph = spaceSceneGraph2D; };
		%enemyFire.setPosition( %object.getPosition() );
		%enemyFire.setSize( "3 1.5" );
		%enemyFire.setGroup( 4 );
		%enemyFire.setLayer( 14 );
		%enemyFire.setImageMap( enemymissileImageMap );
		%enemyFire.setWorldLimit( kill, "-60 -40 60 40" );
		%enemyFire.setCollisionActive( true, true );
		%enemyFire.setCollisionScale("0.9 0.5");
		%enemyFire.setCollisionMaterial( standardMaterial );
		%enemyFire.setCollisionMasks( $playerCollisionGroups, $playerCollisionLayers );
		%enemyFire.setCollisionCallback( true );
		%enemyFire.setLinearVelocityX( -$enemyFireSpeed );
		%enemyFire.tag = "enemyfire";
		
		// Add to Active Enemy Stuff.
		$activeEnemyStuff.add( %enemyFire );
		
		// Missile Fire Audio.
		alxPlay( enemyMissileFireAudio );			
	}
	
	// Reschedule another potential fire.
	%object.schedule( 1000 + mFloor(getRandom()*3000), "enemyFireMissle" );
	
}


//-----------------------------------------------------------------------------
// Boss Fire (Torpedo)
//-----------------------------------------------------------------------------
function fxStaticSprite2D::bossFireTorpedo( %object )
{
	// Is the Player Dead?
	if ( !$playerDead && !$spaceScenePauseFlag )
	{	
		// No, so 
		%angle = %object.bossGun.getRotation();
		// Calculate Fire Vector.
		%fireVec = vectorScale2D(mSin(mDegToRad(%angle)) SPC (-mCos(mDegToRad(%angle))), $bossFireSpeed );
		
		// Create Projectile #1.
		%enemyFire = new fxSceneObject2D() { scenegraph = spaceSceneGraph2D; };
		%enemyFire.setPosition( %object.bossGun.getLinkPoint(%object.bossGun1) );
		%enemyFire.setSize( "1 1" );
		%enemyFire.setGroup( 4 );
		%enemyFire.setLayer( 14 );
		%enemyFire.setWorldLimit( kill, "-52 -40 52 40" );
		%enemyFire.setCollisionActive( true, true );
		%enemyFire.setCollisionMaterial( standardMaterial );
		%enemyFire.setCollisionPolyPrimitive( 6 );
		%enemyFire.setCollisionMasks( $playerCollisionGroups, $playerCollisionLayers );
		%enemyFire.setCollisionCallback( true );
		%enemyFire.setLinearVelocity( %fireVec );
		%enemyFire.tag = "enemyfire";
	
		// Fire FX.
		%fireFX = new fxParticleEffect2D() { scenegraph = spaceSceneGraph2D; };
		%fireFX.loadEffect("~/client/effects/torpedo.eff");
		%fireFX.setLayer( %object.getLayer() );
		%fireFX.setGroup( 14 );
		%fireFX.mount( %enemyFire );
		%fireFX.playEffect();
		
		// Add to Active Enemy Stuff.
		$activeEnemyStuff.add( %enemyFire );

		// Create Projectile #2.
		%enemyFire = new fxSceneObject2D() { scenegraph = spaceSceneGraph2D; };
		%enemyFire.setPosition( %object.bossGun.getLinkPoint(%object.bossGun2) );
		%enemyFire.setSize( "1 1" );
		%enemyFire.setGroup( 4 );
		%enemyFire.setLayer( 14 );
		%enemyFire.setWorldLimit( kill, "-52 -40 52 40" );
		%enemyFire.setCollisionActive( true, true );
		%enemyFire.setCollisionMaterial( standardMaterial );
		%enemyFire.setCollisionPolyPrimitive( 6 );
		%enemyFire.setCollisionMasks( $playerCollisionGroups, $playerCollisionLayers );
		%enemyFire.setCollisionCallback( true );
		%enemyFire.setLinearVelocity( %fireVec );
		%enemyFire.tag = "enemyfire";
	
		// Fire FX.
		%fireFX = new fxParticleEffect2D() { scenegraph = spaceSceneGraph2D; };
		%fireFX.loadEffect("~/client/effects/torpedo.eff");
		%fireFX.setLayer( %object.getLayer() );
		%fireFX.setGroup( 14 );
		%fireFX.mount( %enemyFire );
		%fireFX.playEffect();
		
		// Add to Active Enemy Stuff.
		$activeEnemyStuff.add( %enemyFire );
		
		
		// Torpedo Fire Audio.
		alxPlay( enemyTorpedoFireAudio );		
	}
	
	// Reschedule boss gun-fire. 	
	%object.schedule( $bossFireRate, "bossFireTorpedo" );
}


//-----------------------------------------------------------------------------
// Fast-Forward Enemy Stuff.
//-----------------------------------------------------------------------------
function fastForwardEnemyStuff()
{
	// Fast-Forward anything active.
	for ( %n = 0; %n < $activeEnemyStuff.getCount(); %n++ )
	{
		// Fetch Object.
		%obj = $activeEnemyStuff.getObject(%n);
		
		// Waveform Object?
		if (%obj.tag $= "waveform")
			// Boss?
			if ( %obj.enemyShip.shipType == 3 )
				// Change World Limit to kill it.
				%obj.setWorldLimit( kill, "-100 -100 1000 100" );				
				
		// Send it upwards.
		%obj.setLinearVelocity( "-30 -50" );
	}
}				

